# -------------------------------------------------------------------------------------------------
#  Copyright (C) 2015-2026 Nautech Systems Pty Ltd. All rights reserved.
#  https://nautechsystems.io
#
#  Licensed under the GNU Lesser General Public License Version 3.0 (the "License");
#  You may not use this file except in compliance with the License.
#  You may obtain a copy of the License at https://www.gnu.org/licenses/lgpl-3.0.en.html
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
# -------------------------------------------------------------------------------------------------

# ruff: noqa: UP007

import datetime as dt
from collections.abc import Awaitable
from collections.abc import Callable
from collections.abc import Iterator
from decimal import Decimal
from enum import Enum
from os import PathLike
from typing import Any
from typing import Final
from typing import Union

import numpy as np

from nautilus_trader.core.data import Data

# Python interface type hints
# ---------------------------
# These type definitions provide import resolution and docstrings for Python types
# implemented in PyO3. They will eventually be separated into per-module .pyi files.

###################################################################################################
# Core
###################################################################################################

NAUTILUS_VERSION: Final[str]
NAUTILUS_USER_AGENT: Final[str]

MILLISECONDS_IN_SECOND: Final[int]
NANOSECONDS_IN_SECOND: Final[int]
NANOSECONDS_IN_MILLISECOND: Final[int]
NANOSECONDS_IN_MICROSECOND: Final[int]

class UUID4:
    def __init__(self) -> None: ...
    def __eq__(self, other: object) -> bool: ...
    def __hash__(self) -> int: ...
    @property
    def value(self) -> str: ...
    @staticmethod
    def from_str(value: str) -> UUID4: ...

def secs_to_nanos(secs: float | Decimal) -> int: ...
def secs_to_millis(secs: float | Decimal) -> int: ...
def millis_to_nanos(millis: float | Decimal) -> int: ...
def micros_to_nanos(micros: float | Decimal) -> int: ...
def nanos_to_secs(nanos: int) -> float: ...
def nanos_to_millis(nanos: int) -> int: ...
def nanos_to_micros(nanos: int) -> int: ...
def last_weekday_nanos(year: int, month: int, day: int) -> int: ...
def is_within_last_24_hours(timestamp_ns: int) -> bool: ...
def convert_to_snake_case(input: str) -> str: ...
def is_pycapsule(obj: object) -> bool: ...

###################################################################################################
# Common
###################################################################################################

def get_exchange_rate(
    from_currency: str,
    to_currency: str,
    price_type: PriceType,
    quotes_bid: dict[str, float],
    quotes_ask: dict[str, float],
) -> float | None: ...

# Logging

class LogGuard: ...

def init_logging(
    trader_id: TraderId,
    instance_id: UUID4,
    level_stdout: LogLevel,
    level_file: LogLevel | None = None,
    component_levels: dict[str, str] | None = None,
    directory: str | None = None,
    file_name: str | None = None,
    file_format: str | None = None,
    file_rotate: tuple[int, int] | None = None,
    is_colored: bool | None = None,
    is_bypassed: bool | None = None,
    print_config: bool | None = None,
    log_components_only: bool | None = None,
) -> LogGuard: ...
def tracing_is_initialized() -> bool: ...
def init_tracing() -> None: ...
def log_header(
    trader_id: TraderId,
    machine_id: str,
    instance_id: UUID4,
    component: str,
) -> None: ...
def log_sysinfo(component: str) -> None: ...
def logger_flush() -> None: ...

# Messaging

class BusMessage:
    @property
    def topic(self) -> str: ...
    @property
    def payload(self) -> bytes: ...

class MessageBusListener:
    def __init__(self) -> None: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def close(self) -> None: ...
    def publish(self, topic: str, payload: bytes) -> None: ...
    def stream(self, callback: Callable) -> Awaitable[None]: ...

class Signal:
    def __init__(
        self,
        name: str,
        value: str,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def value(self) -> str: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class CustomData:
    def __init__(
        self,
        data_type: DataType,
        value: bytes,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def data_type(self) -> DataType: ...
    @property
    def value(self) -> str: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class FifoCache:
    def __init__(self) -> None: ...
    def __len__(self) -> int: ...
    def __contains__(self, key: str) -> bool: ...
    @property
    def capacity(self) -> int: ...
    def add(self, key: str) -> None: ...
    def remove(self, key: str) -> None: ...
    def clear(self) -> None: ...

###################################################################################################
# Actor System
###################################################################################################

class DataActor:
    def __init__(self, config: object = None) -> None: ...
    @property
    def actor_id(self) -> ActorId: ...
    @property
    def trader_id(self) -> TraderId | None: ...
    def state(self) -> ComponentState: ...
    def is_ready(self) -> bool: ...
    def is_running(self) -> bool: ...
    def is_stopped(self) -> bool: ...
    def is_disposed(self) -> bool: ...
    def is_degraded(self) -> bool: ...
    def is_faulted(self) -> bool: ...
    def start(self) -> None: ...
    def stop(self) -> None: ...
    def resume(self) -> None: ...
    def reset(self) -> None: ...
    def dispose(self) -> None: ...
    def degrade(self) -> None: ...
    def fault(self) -> None: ...
    def shutdown_system(self, reason: str | None = None) -> None: ...
    def on_signal(self, signal: Signal) -> None: ...
    def on_data(self, data: Any) -> None: ...
    def on_instrument(self, instrument: Any) -> None: ...
    def on_quote(self, tick: QuoteTick) -> None: ...
    def on_trade(self, tick: TradeTick) -> None: ...
    def on_bar(self, bar: Bar) -> None: ...
    def on_book_deltas(self, deltas: OrderBookDeltas) -> None: ...
    def on_book(self, order_book: OrderBook) -> None: ...
    def on_mark_price(self, mark_price: MarkPriceUpdate) -> None: ...
    def on_index_price(self, index_price: IndexPriceUpdate) -> None: ...
    def on_instrument_status(self, data: InstrumentStatus) -> None: ...
    def on_instrument_close(self, update: InstrumentClose) -> None: ...
    def subscribe_data(
        self,
        data_type: DataType,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_instruments(
        self,
        venue: Venue,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_instrument(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_book_deltas(
        self,
        instrument_id: InstrumentId,
        book_type: BookType,
        depth: int | None = None,
        client_id: ClientId | None = None,
        managed: bool = False,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_book_at_interval(
        self,
        instrument_id: InstrumentId,
        book_type: BookType,
        interval_ms: int,
        depth: int | None = None,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_quotes(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_trades(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_mark_prices(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_index_prices(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_bars(
        self,
        bar_type: BarType,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_instrument_status(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def subscribe_instrument_close(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_data(
        self,
        data_type: DataType,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_instruments(
        self,
        venue: Venue,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_instrument(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_quotes(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_trades(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_bars(
        self,
        bar_type: BarType,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_book_deltas(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_book_at_interval(
        self,
        instrument_id: InstrumentId,
        interval_ms: int,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_mark_prices(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_index_prices(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_instrument_status(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def unsubscribe_instrument_close(
        self,
        instrument_id: InstrumentId,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> None: ...
    def request_data(
        self,
        data_type: DataType,
        client_id: ClientId,
        start: int | None = None,
        end: int | None = None,
        limit: int | None = None,
        params: dict[str, str] | None = None,
    ) -> str: ...
    def request_instrument(
        self,
        instrument_id: InstrumentId,
        start: int | None = None,
        end: int | None = None,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> str: ...
    def request_instruments(
        self,
        venue: Venue | None = None,
        start: int | None = None,
        end: int | None = None,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> str: ...
    def request_book_snapshot(
        self,
        instrument_id: InstrumentId,
        depth: int | None = None,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> str: ...
    def request_quotes(
        self,
        instrument_id: InstrumentId,
        start: int | None = None,
        end: int | None = None,
        limit: int | None = None,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> str: ...
    def request_trades(
        self,
        instrument_id: InstrumentId,
        start: int | None = None,
        end: int | None = None,
        limit: int | None = None,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> str: ...
    def request_bars(
        self,
        bar_type: BarType,
        start: int | None = None,
        end: int | None = None,
        limit: int | None = None,
        client_id: ClientId | None = None,
        params: dict[str, str] | None = None,
    ) -> str: ...

###################################################################################################
# Cryptography
###################################################################################################

def mask_api_key(api_key: str) -> str: ...
def hmac_signature(secret: str, data: str) -> str: ...
def rsa_signature(private_key_pem: str, data: str) -> str: ...
def ed25519_signature(private_key: bytes, data: str) -> str: ...

###################################################################################################
# Model
###################################################################################################

HIGH_PRECISION: Final[bool]
FIXED_SCALAR: Final[float]
FIXED_PRECISION: Final[int]
PRECISION_BYTES: Final[int]

class DataType:
    def __init__(self, type_name: str, metadata: dict[str, str] | None = None) -> None: ...
    @property
    def type_name(self) -> str: ...
    @property
    def metadata(self) -> dict[str, str] | None: ...
    @property
    def topic(self) -> str: ...

# Accounting

class Position:
    def __init__(
        self,
        instrument: CurrencyPair | CryptoPerpetual | Equity | OptionContract | SyntheticInstrument,
        fill: OrderFilled,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> Position: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def id(self) -> PositionId: ...
    @property
    def symbol(self) -> Symbol: ...
    @property
    def venue(self) -> Venue: ...
    @property
    def opening_order_id(self) -> ClientOrderId: ...
    @property
    def closing_order_id(self) -> ClientOrderId | None: ...
    @property
    def quantity(self) -> Quantity: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def peak_qty(self) -> Quantity: ...
    @property
    def signed_qty(self) -> float: ...
    @property
    def entry(self) -> OrderSide: ...
    @property
    def side(self) -> PositionSide: ...
    @property
    def ts_opened(self) -> int: ...
    @property
    def duration_ns(self) -> int: ...
    @property
    def avg_px_open(self) -> Price: ...
    @property
    def event_count(self) -> int: ...
    @property
    def venue_order_ids(self) -> list[VenueOrderId]: ...
    @property
    def client_order_ids(self) -> list[ClientOrderId]: ...
    @property
    def trade_ids(self) -> list[TradeId]: ...
    @property
    def last_trade_id(self) -> TradeId | None: ...
    @property
    def events(self) -> list[OrderFilled]: ...
    @property
    def adjustments(self) -> list[PositionAdjusted]: ...
    @property
    def is_open(self) -> bool: ...
    @property
    def is_closed(self) -> bool: ...
    @property
    def is_long(self) -> bool: ...
    @property
    def is_short(self) -> bool: ...
    @property
    def realized_return(self) -> float: ...
    @property
    def realized_pnl(self) -> Money | None: ...
    @property
    def ts_closed(self) -> int | None: ...
    @property
    def avg_px_close(self) -> Price | None: ...
    def unrealized_pnl(self, price: Price) -> Money: ...
    def total_pnl(self, price: Price) -> Money: ...
    def commissions(self) -> list[Money]: ...
    def apply(self, fill: OrderFilled) -> None: ...
    def apply_adjustment(self, adjustment: PositionAdjusted) -> None: ...
    def is_opposite_side(self, side: OrderSide) -> bool: ...
    def calculate_pnl(
        self, avg_px_open: float, avg_px_close: float, quantity: Quantity
    ) -> Money: ...
    def notional_value(self, price: Price) -> Money: ...

class MarginAccount:
    def __init__(
        self,
        event: AccountState,
        calculate_account_state: bool,
    ) -> None: ...
    @property
    def id(self) -> AccountId: ...
    @property
    def default_leverage(self) -> float: ...
    def leverages(self) -> dict[InstrumentId, float]: ...
    def leverage(self, instrument_id: InstrumentId) -> float: ...
    def set_default_leverage(self, leverage: float) -> None: ...
    def set_leverage(self, instrument_id: InstrumentId, leverage: float) -> None: ...
    def is_unleveraged(self) -> bool: ...
    def update_initial_margin(self, instrument_id: InstrumentId, initial_margin: Money) -> None: ...
    def initial_margin(self, instrument_id: InstrumentId) -> Money: ...
    def initial_margins(self) -> dict[InstrumentId, Money]: ...
    def update_maintenance_margin(
        self, instrument_id: InstrumentId, maintenance_margin: Money
    ) -> None: ...
    def maintenance_margin(self, instrument_id: InstrumentId) -> Money: ...
    def maintenance_margins(self) -> dict[InstrumentId, Money]: ...
    def calculate_initial_margin(
        self,
        instrument: Instrument,
        quantity: Quantity,
        price: Price,
        use_quote_for_inverse: bool | None = None,
    ) -> Money: ...
    def calculate_maintenance_margin(
        self,
        instrument: Instrument,
        quantity: Quantity,
        price: Price,
        use_quote_for_inverse: bool | None = None,
    ) -> Money: ...

class CashAccount:
    def __init__(
        self,
        event: AccountState,
        calculate_account_state: bool,
    ) -> None: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def id(self) -> AccountId: ...
    @property
    def base_currency(self) -> Currency | None: ...
    @property
    def last_event(self) -> AccountState | None: ...
    def events(self) -> list[AccountState]: ...
    @property
    def event_count(self) -> int: ...
    def balance_total(self, currency: Currency | None) -> Money | None: ...
    def balances_total(self) -> dict[Currency, Money]: ...
    def balance_free(self, currency: Currency | None) -> Money | None: ...
    def balances_free(self) -> dict[Currency, Money]: ...
    def balance_locked(self, currency: Currency | None) -> Money | None: ...
    def balances_locked(self) -> dict[Currency, Money]: ...
    def apply(self, event: AccountState) -> None: ...
    def calculate_balance_locked(
        self,
        instrument: Instrument,
        side: OrderSide,
        quantity: Quantity,
        price: Price,
        use_quote_for_inverse: bool | None = None,
    ) -> Money: ...
    def calculate_commission(
        self,
        instrument: Instrument,
        last_qty: Quantity,
        last_px: Price,
        liquidity_side: LiquiditySide,
        use_quote_for_inverse: bool | None = None,
    ) -> Money: ...
    def calculate_pnls(
        self,
        instrument: Instrument,
        fill: OrderFilled,
        position: Position | None = None,
    ) -> list[Money]: ...

type Account = Union[CashAccount, MarginAccount]

# Accounting transformers

def cash_account_from_account_events(
    events: list[dict],
    calculate_account_state: bool,
) -> CashAccount: ...
def margin_account_from_account_events(
    events: list[dict],
    calculate_account_state: bool,
) -> MarginAccount: ...

# Data types

def drop_cvec_pycapsule(capsule: object) -> None: ...

class BarSpecification:
    def __init__(
        self,
        step: int,
        aggregation: BarAggregation,
        price_type: PriceType,
    ) -> None: ...
    @property
    def step(self) -> int: ...
    @property
    def aggregation(self) -> BarAggregation: ...
    @property
    def price_type(self) -> PriceType: ...
    @property
    def timedelta(self) -> dt.timedelta: ...
    @staticmethod
    def fully_qualified_name() -> str: ...

class BarType:
    def __init__(
        self,
        instrument_id: InstrumentId,
        spec: BarSpecification,
        aggregation_source: AggregationSource | None = None,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def spec(self) -> BarSpecification: ...
    @property
    def aggregation_source(self) -> AggregationSource: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def from_str(value: str) -> BarType: ...
    @staticmethod
    def new_composite(
        instrument_id: InstrumentId,
        spec: BarSpecification,
        aggregation_source: AggregationSource,
    ) -> BarType: ...
    def is_standard(self) -> bool: ...
    def is_composite(self) -> bool: ...
    def standard(self) -> BarType: ...
    def composite(self) -> BarType: ...
    def id_spec_key(self) -> str: ...

class Bar:
    def __init__(
        self,
        bar_type: BarType,
        open: Price,
        high: Price,
        low: Price,
        close: Price,
        volume: Quantity,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def bar_type(self) -> BarType: ...
    @property
    def open(self) -> Price: ...
    @property
    def high(self) -> Price: ...
    @property
    def low(self) -> Price: ...
    @property
    def close(self) -> Price: ...
    @property
    def volume(self) -> Quantity: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> Bar: ...
    @staticmethod
    def from_json(data: bytes) -> Bar: ...
    @staticmethod
    def from_msgpack(data: bytes) -> Bar: ...
    def as_pycapsule(self) -> object: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class Bet:
    def __init__(self, price: Decimal, stake: Decimal, side: BetSide) -> None: ...
    @property
    def price(self) -> Decimal: ...
    @property
    def stake(self) -> Decimal: ...
    @property
    def side(self) -> BetSide: ...
    @staticmethod
    def from_stake_or_liability(price: float, volume: float, side: BetSide) -> Bet: ...
    @staticmethod
    def from_stake(price: float, stake: float, side: BetSide) -> Bet: ...
    @staticmethod
    def from_liability(price: float, liability: float, side: BetSide) -> Bet: ...
    def exposure(self) -> Decimal: ...
    def liability(self) -> Decimal: ...
    def profit(self) -> Decimal: ...
    def outcome_win_payoff(self) -> Decimal: ...
    def outcome_lose_payoff(self) -> Decimal: ...
    def hedging_stake(self, price: Decimal) -> Decimal: ...
    def hedging_bet(self, price: Decimal) -> Bet: ...

class BetPosition:
    def __init__(self) -> None: ...
    @property
    def price(self) -> Decimal: ...
    @property
    def side(self) -> BetSide | None: ...
    @property
    def exposure(self) -> Decimal: ...
    @property
    def realized_pnl(self) -> Decimal: ...
    def add_bet(self, bet: Bet) -> None: ...
    def as_bet(self) -> Bet | None: ...
    def unrealized_pnl(self, price: Decimal) -> Decimal: ...
    def total_pnl(self, price: Decimal) -> Decimal: ...
    def flattening_bet(self, price: Decimal) -> Bet | None: ...
    def reset(self) -> None: ...

def calc_bets_pnl(bets: list[Bet]) -> Decimal: ...
def probability_to_bet(probability: Decimal, volume: Decimal, side: OrderSide) -> Bet: ...
def inverse_probability_to_bet(probability: Decimal, volume: Decimal, side: OrderSide) -> Bet: ...

class BookOrder:
    def __init__(
        self,
        side: OrderSide,
        price: Price,
        size: Quantity,
        order_id: int,
    ) -> None: ...
    @property
    def side(self) -> OrderSide: ...
    @property
    def price(self) -> Price: ...
    @property
    def size(self) -> Quantity: ...
    @property
    def order_id(self) -> int: ...

class OrderBookDelta:
    def __init__(
        self,
        instrument_id: InstrumentId,
        action: BookAction,
        order: BookOrder | None,
        flags: int,
        sequence: int,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def action(self) -> BookAction: ...
    @property
    def order(self) -> BookOrder: ...
    @property
    def flags(self) -> int: ...
    @property
    def sequence(self) -> int: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> OrderBookDelta: ...
    @staticmethod
    def from_json(data: bytes) -> OrderBookDelta: ...
    @staticmethod
    def from_msgpack(data: bytes) -> OrderBookDelta: ...
    def as_pycapsule(self) -> object: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class OrderBookDeltas:
    def __init__(
        self,
        instrument_id: InstrumentId,
        deltas: list[OrderBookDelta],
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def deltas(self) -> list[OrderBookDelta]: ...
    @property
    def flags(self) -> int: ...
    @property
    def sequence(self) -> int: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...

class OrderBookDepth10:
    def __init__(
        self,
        instrument_id: InstrumentId,
        bids: list[BookOrder],
        asks: list[BookOrder],
        bid_counts: list[int],
        ask_counts: list[int],
        flags: int,
        sequence: int,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def bids(self) -> list[BookOrder]: ...
    @property
    def asks(self) -> list[BookOrder]: ...
    @property
    def bid_counts(self) -> list[int]: ...
    @property
    def ask_counts(self) -> list[int]: ...
    @property
    def flags(self) -> int: ...
    @property
    def sequence(self) -> int: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def get_stub() -> OrderBookDepth10: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> OrderBookDepth10: ...
    @staticmethod
    def from_json(data: bytes) -> OrderBookDepth10: ...
    @staticmethod
    def from_msgpack(data: bytes) -> OrderBookDepth10: ...
    def as_pycapsule(self) -> object: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class QuoteTick:
    def __init__(
        self,
        instrument_id: InstrumentId,
        bid_price: Price,
        ask_price: Price,
        bid_size: Quantity,
        ask_size: Quantity,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def bid_price(self) -> Price: ...
    @property
    def ask_price(self) -> Price: ...
    @property
    def bid_size(self) -> Quantity: ...
    @property
    def ask_size(self) -> Quantity: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> QuoteTick: ...
    @staticmethod
    def from_json(data: bytes) -> QuoteTick: ...
    @staticmethod
    def from_msgpack(data: bytes) -> QuoteTick: ...
    def extract_price(self) -> Price: ...
    def extract_size(self) -> Quantity: ...
    def as_pycapsule(self) -> object: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class TradeTick:
    def __init__(
        self,
        instrument_id: InstrumentId,
        price: Price,
        size: Quantity,
        aggressor_side: AggressorSide,
        trade_id: TradeId,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def price(self) -> Price: ...
    @property
    def size(self) -> Quantity: ...
    @property
    def aggressor_side(self) -> AggressorSide: ...
    @property
    def trade_id(self) -> TradeId: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> TradeTick: ...
    @staticmethod
    def from_json(data: bytes) -> TradeTick: ...
    @staticmethod
    def from_msgpack(data: bytes) -> TradeTick: ...
    def as_pycapsule(self) -> object: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class MarkPriceUpdate:
    def __init__(
        self,
        instrument_id: InstrumentId,
        value: Price,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def value(self) -> Price: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> MarkPriceUpdate: ...
    @staticmethod
    def from_json(data: bytes) -> MarkPriceUpdate: ...
    @staticmethod
    def from_msgpack(data: bytes) -> MarkPriceUpdate: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class IndexPriceUpdate:
    def __init__(
        self,
        instrument_id: InstrumentId,
        value: Price,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def value(self) -> Price: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> IndexPriceUpdate: ...
    @staticmethod
    def from_json(data: bytes) -> IndexPriceUpdate: ...
    @staticmethod
    def from_msgpack(data: bytes) -> IndexPriceUpdate: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class InstrumentClose:
    def __init__(
        self,
        instrument_id: InstrumentId,
        close_price: Price,
        close_type: InstrumentCloseType,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def close_price(self) -> Price: ...
    @property
    def close_type(self) -> InstrumentCloseType: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata() -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> InstrumentClose: ...
    @staticmethod
    def from_json(data: bytes) -> InstrumentClose: ...
    @staticmethod
    def from_msgpack(data: bytes) -> InstrumentClose: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json_bytes(self) -> bytes: ...
    def to_msgpack_bytes(self) -> bytes: ...

class FundingRateUpdate:
    def __init__(
        self,
        instrument_id: InstrumentId,
        rate: Decimal,
        ts_event: int,
        ts_init: int,
        next_funding_ns: int | None = None,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def rate(self) -> Decimal: ...
    @property
    def next_funding_ns(self) -> int | None: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @staticmethod
    def fully_qualified_name() -> str: ...
    @staticmethod
    def get_metadata(instrument_id: InstrumentId) -> dict[str, str]: ...
    @staticmethod
    def get_fields() -> dict[str, str]: ...
    @staticmethod
    def from_dict(values: dict[str, Any]) -> FundingRateUpdate: ...
    @staticmethod
    def from_json(data: bytes) -> FundingRateUpdate: ...
    @staticmethod
    def from_msgpack(data: bytes) -> FundingRateUpdate: ...
    def to_dict(self) -> dict[str, Any]: ...
    def to_json(self) -> bytes: ...
    def to_msgpack(self) -> bytes: ...

class InstrumentStatus:
    def __init__(
        self,
        instrument_id: InstrumentId,
        action: MarketStatusAction,
        ts_event: int,
        ts_init: int,
        reason: str | None,
        trading_event: str | None,
        is_trading: bool | None,
        is_quoting: bool | None,
        is_short_sell_restricted: bool | None,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def action(self) -> MarketStatusAction: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reason(self) -> str | None: ...
    @property
    def trading_event(self) -> str | None: ...
    @property
    def is_trading(self) -> bool | None: ...
    @property
    def is_quoting(self) -> bool | None: ...
    @property
    def is_short_sell_restricted(self) -> bool | None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> InstrumentStatus: ...
    def to_dict(self) -> dict[str, str]: ...

# Enums

class AccountType(Enum):
    CASH = "CASH"
    MARGIN = "MARGIN"
    BETTING = "BETTING"

class AggregationSource(Enum):
    EXTERNAL = "EXTERNAL"
    INTERNAL = "INTERNAL"

class AggressorSide(Enum):
    BUYER = "BUYER"
    SELLER = "SELLER"

class AssetClass(Enum):
    FX = "FX"
    EQUITY = "EQUITY"
    COMMODITY = "COMMODITY"
    DEBT = "DEBT"
    INDEX = "INDEX"
    CRYPTOCURRENCY = "CRYPTOCURRENCY"
    ALTERNATIVE = "ALTERNATIVE"

class InstrumentClass(Enum):
    SPOT = "SPOT"
    SWAP = "SWAP"
    FUTURE = "FUTURE"
    FUTURES_SPREAD = "FUTURES_SPREAD"
    FORWARD = "FORWARD"
    CFD = "CFD"
    BOND = "BOND"
    OPTION = "OPTION"
    OPTION_SPREAD = "OPTION_SPEAD"
    WARRANT = "WARRANT"
    SPORTS_BETTING = "SPORTS_BETTING"
    BINARY_OPTION = "BINARY_OPTION"

class BarAggregation(Enum):
    TICK = "TICK"
    TICK_IMBALANCE = "TICK_IMBALANCE"
    TICK_RUNS = "TICK_RUNS"
    VOLUME = "VOLUME"
    VOLUME_IMBALANCE = "VOLUME_IMBALANCE"
    VOLUME_RUNS = "VOLUME_RUNS"
    VALUE = "VALUE"
    VALUE_IMBALANCE = "VALUE_IMBALANCE"
    VALUE_RUNS = "VALUE_RUNS"
    MILLISECOND = "MILLISECOND"
    SECOND = "SECOND"
    MINUTE = "MINUTE"
    HOUR = "HOUR"
    DAY = "DAY"
    WEEK = "WEEK"
    MONTH = "MONTH"
    YEAR = "YEAR"
    RENKO = "RENKO"

class BarIntervalType(Enum):
    LEFT_OPEN = "LEFT_OPEN"
    RIGHT_OPEN = "RIGHT_OPEN"

class BetSide(Enum):
    BACK = "BACK"
    LAY = "LAY"

    @staticmethod
    def from_order_side(order_side: OrderSide) -> BetSide: ...
    def opposite(self) -> BetSide: ...

class BookAction(Enum):
    ADD = "ADD"
    UPDATE = "UPDATE"
    DELETE = "DELETE"
    CLEAR = "CLEAR"

class BookType(Enum):
    L1_MBP = "L1_MBP"
    L2_MBP = "L2_MBP"
    L3_MBO = "L3_MBO"

class ContingencyType(Enum):
    NO_CONTINGENCY = "NO_CONTINGENCY"
    OCO = "OCO"
    OTO = "OTO"
    OUO = "OUO"

class ComponentState(Enum):
    PRE_INITIALIZED = "PRE_INITIALIZED"
    READY = "READY"
    STARTING = "STARTING"
    RUNNING = "RUNNING"
    STOPPING = "STOPPING"
    STOPPED = "STOPPED"
    RESUMING = "RESUMING"
    RESETTING = "RESETTING"
    DISPOSING = "DISPOSING"
    DISPOSED = "DISPOSED"
    DEGRADING = "DEGRADING"
    DEGRADED = "DEGRADED"
    FAULTING = "FAULTING"
    FAULTED = "FAULTED"

class CurrencyType(Enum):
    CRYPTO = "CRYPTO"
    FIAT = "FIAT"
    COMMODITY_BACKED = "COMMODITY_BACKED"
    @classmethod
    def from_str(cls, value: str) -> CurrencyType: ...

class Environment(Enum):
    BACKTEST = "BACKTEST"
    SANDBOX = "SANDBOX"
    LIVE = "LIVE"
    @classmethod
    def from_str(cls, value: str) -> Environment: ...

class InstrumentCloseType(Enum):
    END_OF_SESSION = "END_OF_SESSION"
    CONTRACT_EXPIRED = "CONTRACT_EXPIRED"

class LiquiditySide(Enum):
    MAKER = "MAKER"
    TAKER = "TAKER"
    NO_LIQUIDITY_SIDE = "NO_LIQUIDITY_SIDE"

class MarketStatus(Enum):
    OPEN = "OPEN"
    CLOSED = "CLOSED"
    PAUSED = "PAUSED"
    SUSPENDED = "SUSPENDED"
    NOT_AVAILABLE = "NOT_AVAILABLE"

class MarketStatusAction(Enum):
    NONE = "NONE"
    PRE_OPEN = "PRE_OPEN"
    PRE_CROSS = "PRE_CROSS"
    QUOTING = "QUOTING"
    CROSS = "CROSS"
    ROTATION = "ROTATION"
    NEW_PRICE_INDICATION = "NEW_PRICE_INDICATION"
    TRADING = "TRADING"
    HALT = "HALT"
    PAUSE = "PAUSE"
    SUSPEND = "SUSPEND"
    PRE_CLOSE = "PRE_CLOSE"
    CLOSE = "CLOSE"
    POST_CLOSE = "POST_CLOSE"
    SHORT_SELL_RESTRICTION_CHANGE = "SHORT_SELL_RESTRICTION_CHANGE"
    NOT_AVAILABLE_FOR_TRADING = "NOT_AVAILABLE_FOR_TRADING"

class OmsType(Enum):
    UNSPECIFIED = "UNSPECIFIED"
    NETTING = "NETTING"
    HEDGING = "HEDGING"

class OptionKind(Enum):
    CALL = "CALL"
    PUT = "PUT"

class OrderSide(Enum):
    NO_ORDER_SIDE = "NO_ORDER_SIDE"
    BUY = "BUY"
    SELL = "SELL"

    @classmethod
    def from_str(cls, value: str) -> OrderSide: ...

class OrderStatus(Enum):
    INITIALIZED = "INITIALIZED"
    DENIED = "DENIED"
    EMULATED = "EMULATED"
    RELEASED = "RELEASED"
    SUBMITTED = "SUBMITTED"
    ACCEPTED = "ACCEPTED"
    REJECTED = "REJECTED"
    CANCELED = "CANCELED"
    EXPIRED = "EXPIRED"
    TRIGGERED = "TRIGGERED"
    PENDING_UPDATE = "PENDING_UPDATE"
    PENDING_CANCEL = "PENDING_CANCEL"
    PARTIALLY_FILLED = "PARTIALLY_FILLED"
    FILLED = "FILLED"

class OrderType(Enum):
    MARKET = "MARKET"
    LIMIT = "LIMIT"
    STOP_MARKET = "STOP_MARKET"
    STOP_LIMIT = "STOP_LIMIT"
    MARKET_TO_LIMIT = "MARKET_TO_LIMIT"
    MARKET_IF_TOUCHED = "MARKET_IF_TOUCHED"
    LIMIT_IF_TOUCHED = "LIMIT_IF_TOUCHED"
    TRAILING_STOP_MARKET = "TRAILING_STOP_MARKET"
    TRAILING_STOP_LIMIT = "TRAILING_STOP_LIMIT"

class PositionAdjustmentType(Enum):
    COMMISSION = "COMMISSION"
    FUNDING = "FUNDING"

class PositionSide(Enum):
    FLAT = "FLAT"
    LONG = "LONG"
    SHORT = "SHORT"

class PriceType(Enum):
    BID = "BID"
    ASK = "ASK"
    MID = "MID"
    LAST = "LAST"
    MARK = "MARK"

class RecordFlag(Enum):
    F_LAST = "F_LAST"
    F_TOB = "F_TOB"
    F_SNAPSHOT = "F_SNAPSHOT"
    F_MBP = "F_MBP"

class TimeInForce(Enum):
    GTC = "GTC"
    IOC = "IOC"
    FOK = "FOK"
    GTD = "GTD"
    DAY = "DAY"
    AT_THE_OPEN = "AT_THE_OPEN"
    AT_THE_CLOSE = "AT_THE_CLOSE"

class TradingState(Enum):
    ACTIVE = "ACTIVE"
    HALTED = "HALTED"
    REDUCING = "REDUCING"

class TrailingOffsetType(Enum):
    PRICE = "PRICE"
    BASIS_POINTS = "BASIS_POINTS"
    TICKS = "TICKS"
    PRICE_TIER = "PRICE_TIER"

class TriggerType(Enum):
    DEFAULT = "DEFAULT"
    LAST_PRICE = "LAST_PRICE"
    MARK_PRICE = "MARK_PRICE"
    INDEX_PRICE = "INDEX_PRICE"
    BID_ASK = "BID_ASK"
    DOUBLE_LAST = "DOUBLE_LAST"
    DOUBLE_BID_ASK = "DOUBLE_BID_ASK"
    LAST_OR_BID_ASK = "LAST_OR_BID_ASK"
    MID_POINT = "MID_POINT"

class MovingAverageType(Enum):
    SIMPLE = "SIMPLE"
    EXPONENTIAL = "EXPONENTIAL"
    DOUBLE_EXPONENTIAL = "DOUBLE_EXPONENTIAL"
    WILDER = "WILDER"
    HULL = "HULL"
    WEIGHTED = "WEIGHTED"
    VARIABLE_INDEX_DYNAMIC = "VARIABLE_INDEX_DYNAMIC"

class LogLevel(Enum):
    OFF = "OFF"
    DEBUG = "DEBUG"
    INFO = "INFO"
    WARNING = "WARNING"
    ERROR = "ERROR"

class LogColor(Enum):
    NORMAL = "NORMAL"
    GREEN = "GREEN"
    BLUE = "BLUE"
    MAGENTA = "MAGENTA"
    CYAN = "CYAN"
    YELLOW = "YELLOW"
    RED = "RED"

class ForexSession(Enum):
    SYDNEY = "SYDNEY"
    TOKYO = "TOKYO"
    LONDON = "LONDON"
    NEW_YORK = "NEW_YORK"

# Identifiers

class AccountId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> AccountId: ...
    @property
    def value(self) -> str: ...

class ActorId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> ActorId: ...
    @property
    def value(self) -> str: ...

class ClientId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> ClientId: ...
    @property
    def value(self) -> str: ...

class ClientOrderId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> ClientOrderId: ...
    @property
    def value(self) -> str: ...

class ComponentId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> ComponentId: ...
    @property
    def value(self) -> str: ...

class ExecAlgorithmId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> ExecAlgorithmId: ...
    @property
    def value(self) -> str: ...

class InstrumentId:
    def __init__(self, symbol: Symbol, venue: Venue) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> InstrumentId: ...
    @property
    def symbol(self) -> Symbol: ...
    @property
    def venue(self) -> Venue: ...
    @property
    def value(self) -> str: ...

class OrderListId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> OrderListId: ...
    @property
    def value(self) -> str: ...

class PositionId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> PositionId: ...
    @property
    def value(self) -> str: ...

class StrategyId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> StrategyId: ...
    @property
    def value(self) -> str: ...

class Symbol:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> Symbol: ...
    @property
    def value(self) -> str: ...
    @property
    def is_composite(self) -> bool: ...
    @property
    def root(self) -> str: ...
    @property
    def topic(self) -> str: ...

class TradeId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> TradeId: ...
    @property
    def value(self) -> str: ...

class TraderId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> TraderId: ...
    @property
    def value(self) -> str: ...

class Venue:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> Venue: ...
    @property
    def value(self) -> str: ...

class VenueOrderId:
    def __init__(self, value: str) -> None: ...
    @classmethod
    def from_str(cls, value: str) -> VenueOrderId: ...
    @property
    def value(self) -> str: ...

# Orders

class LimitOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        price: Price,
        time_in_force: TimeInForce,
        post_only: bool,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        display_qty: Quantity | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ): ...
    @classmethod
    def create(cls, init: OrderInitialized) -> LimitOrder: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def order_type(self) -> OrderType: ...
    @property
    def side(self) -> OrderSide: ...
    @property
    def quantity(self) -> Quantity: ...
    @property
    def price(self) -> Price: ...
    @property
    def expire_time(self) -> int | None: ...
    @property
    def status(self) -> OrderStatus: ...
    @property
    def time_in_force(self) -> TimeInForce: ...
    @property
    def is_post_only(self) -> bool: ...
    @property
    def is_reduce_only(self) -> bool: ...
    @property
    def is_quote_quantity(self) -> bool: ...
    @property
    def has_price(self) -> bool: ...
    @property
    def has_trigger_price(self) -> bool: ...
    @property
    def is_passive(self) -> bool: ...
    @property
    def is_aggressive(self) -> bool: ...
    @property
    def is_open(self) -> bool: ...
    @property
    def is_closed(self) -> bool: ...
    @property
    def is_emulated(self) -> bool: ...
    @property
    def is_active_local(self) -> bool: ...
    @property
    def is_primary(self) -> bool: ...
    @property
    def is_spawned(self) -> bool: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> LimitOrder: ...
    def apply(self, event: object) -> None: ...

class LimitIfTouchedOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        price: Price,
        trigger_price: Price,
        trigger_type: TriggerType,
        time_in_force: TimeInForce,
        post_only: bool,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        display_qty: Quantity | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ) -> None: ...
    @classmethod
    def create(cls, init: OrderInitialized) -> LimitIfTouchedOrder: ...
    def apply(self, event: object) -> None: ...

class MarketOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        init_id: UUID4,
        ts_init: int,
        time_in_force: TimeInForce,
        reduce_only: bool,
        quote_quantity: bool,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ) -> None: ...
    @classmethod
    def create(cls, init: OrderInitialized) -> MarketOrder: ...
    def to_dict(self) -> dict[str, str]: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> MarketOrder: ...
    @staticmethod
    def opposite_side(side: OrderSide) -> OrderSide: ...
    @staticmethod
    def closing_side(side: PositionSide) -> OrderSide: ...
    def signed_decimal_qty(self) -> Decimal: ...
    def would_reduce_only(self, side: PositionSide, position_qty: Quantity) -> bool: ...
    def commission(self, currency: Currency) -> Money | None: ...
    def commissions(self) -> dict[Currency, Money]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def position_id(self) -> PositionId | None: ...
    @property
    def last_trade_id(self) -> TradeId | None: ...
    @property
    def quantity(self) -> Quantity: ...
    @property
    def side(self) -> OrderSide: ...
    @property
    def order_type(self) -> OrderType: ...
    @property
    def price(self) -> Price | None: ...
    def apply(self, event: object) -> None: ...

class MarketToLimitOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        time_in_force: TimeInForce,
        post_only: bool,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        display_qty: Quantity | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ): ...
    @classmethod
    def create(cls, init: OrderInitialized) -> MarketToLimitOrder: ...
    def apply(self, event: object) -> None: ...

class MarketIfTouchedOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        trigger_price: Price,
        trigger_type: TriggerType,
        time_in_force: TimeInForce,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ): ...
    @classmethod
    def create(cls, init: OrderInitialized) -> MarketIfTouchedOrder: ...
    def apply(self, event: object) -> None: ...

class StopLimitOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        price: Price,
        trigger_price: Price,
        trigger_type: TriggerType,
        time_in_force: TimeInForce,
        post_only: bool,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        display_qty: Quantity | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ): ...
    @classmethod
    def create(cls, init: OrderInitialized) -> StopLimitOrder: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> StopLimitOrder: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def order_type(self) -> OrderType: ...
    @property
    def side(self) -> OrderSide: ...
    @property
    def quantity(self) -> Quantity: ...
    @property
    def price(self) -> Price: ...
    @property
    def trigger_price(self) -> Price: ...
    @property
    def trigger_type(self) -> TriggerType: ...
    @property
    def time_in_force(self) -> TimeInForce: ...
    @property
    def is_post_only(self) -> bool: ...
    @property
    def is_reduce_only(self) -> bool: ...
    @property
    def is_quote_quantity(self) -> bool: ...
    @property
    def is_passive(self) -> bool: ...
    @property
    def is_aggressive(self) -> bool: ...
    @property
    def is_closed(self) -> bool: ...
    @property
    def is_open(self) -> bool: ...
    @property
    def status(self) -> OrderStatus: ...
    @property
    def has_price(self) -> bool: ...
    @property
    def has_trigger_price(self) -> bool: ...
    @property
    def expire_time(self) -> int | None: ...
    def apply(self, event: object) -> None: ...

class StopMarketOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        trigger_price: Price,
        trigger_type: TriggerType,
        time_in_force: TimeInForce,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        display_qty: Quantity | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ): ...
    @classmethod
    def create(cls, init: OrderInitialized) -> StopMarketOrder: ...
    def apply(self, event: object) -> None: ...

class TrailingStopLimitOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        price: Price,
        trigger_price: Price,
        trigger_type: TriggerType,
        limit_offset: Decimal,
        trailing_offset: Decimal,
        trailing_offset_type: TrailingOffsetType,
        time_in_force: TimeInForce,
        post_only: bool,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        display_qty: Quantity | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ): ...
    @classmethod
    def create(cls, init: OrderInitialized) -> TrailingStopLimitOrder: ...
    def apply(self, event: object) -> None: ...

class TrailingStopMarketOrder:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        quantity: Quantity,
        trigger_price: Price,
        trigger_type: TriggerType,
        trailing_offset: Decimal,
        trailing_offset_type: TrailingOffsetType,
        time_in_force: TimeInForce,
        reduce_only: bool,
        quote_quantity: bool,
        init_id: UUID4,
        ts_init: int,
        expire_time: int | None = None,
        display_qty: Quantity | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ): ...
    @classmethod
    def create(cls, init: OrderInitialized) -> TrailingStopMarketOrder: ...
    def apply(self, event: object) -> None: ...

type Order = Union[
    LimitOrder,
    LimitIfTouchedOrder,
    MarketOrder,
    MarketToLimitOrder,
    MarketIfTouchedOrder,
    StopLimitOrder,
    StopMarketOrder,
    TrailingStopLimitOrder,
    TrailingStopMarketOrder,
]

# Objects

class Currency:
    def __init__(
        self,
        code: str,
        precision: int,
        iso4217: int,
        name: str,
        currency_type: CurrencyType,
    ) -> None: ...
    @property
    def code(self) -> str: ...
    @property
    def precision(self) -> int: ...
    @property
    def iso4217(self) -> int: ...
    @property
    def name(self) -> str: ...
    @property
    def currency_type(self) -> CurrencyType: ...
    @staticmethod
    def is_fiat(code: str) -> bool: ...
    @staticmethod
    def is_crypto(code: str) -> bool: ...
    @staticmethod
    def is_commodity_backed(code: str) -> bool: ...
    @staticmethod
    def from_str(value: str, strict: bool = False) -> Currency: ...
    @staticmethod
    def register(currency: Currency, overwrite: bool = False) -> None: ...

class Money:
    def __init__(self, value: float, currency: Currency) -> None: ...
    @property
    def raw(self) -> int: ...
    @property
    def currency(self) -> Currency: ...
    @staticmethod
    def zero(currency: Currency) -> Money: ...
    @staticmethod
    def from_raw(raw: int, currency: Currency) -> Money: ...
    @staticmethod
    def from_str(value: str) -> Money: ...
    def is_zero(self) -> bool: ...
    def as_decimal(self) -> Decimal: ...
    def as_double(self) -> float: ...
    def to_formatted_str(self) -> str: ...

class Price:
    def __init__(self, value: float, precision: int) -> None: ...
    @property
    def raw(self) -> int: ...
    @property
    def precision(self) -> int: ...
    @staticmethod
    def from_raw(raw: int, precision: int) -> Price: ...
    @staticmethod
    def zero(precision: int = 0) -> Price: ...
    @staticmethod
    def from_int(value: int) -> Price: ...
    @staticmethod
    def from_str(value: str) -> Price: ...
    @staticmethod
    def from_decimal(value: Decimal) -> Price: ...
    @staticmethod
    def from_decimal_dp(value: Decimal, precision: int) -> Price: ...
    def is_zero(self) -> bool: ...
    def is_positive(self) -> bool: ...
    def as_double(self) -> float: ...
    def as_decimal(self) -> Decimal: ...
    def to_formatted_str(self) -> str: ...

class Quantity:
    def __init__(self, value: float, precision: int) -> None: ...
    @property
    def raw(self) -> int: ...
    @property
    def precision(self) -> int: ...
    @staticmethod
    def from_raw(raw: int, precision: int) -> Quantity: ...
    @staticmethod
    def zero(precision: int = 0) -> Quantity: ...
    @staticmethod
    def from_int(value: int) -> Quantity: ...
    @staticmethod
    def from_str(value: str) -> Quantity: ...
    @staticmethod
    def from_decimal(value: Decimal) -> Quantity: ...
    @staticmethod
    def from_decimal_dp(value: Decimal, precision: int) -> Quantity: ...
    def is_zero(self) -> bool: ...
    def is_positive(self) -> bool: ...
    def as_decimal(self) -> Decimal: ...
    def as_double(self) -> float: ...
    def to_formatted_str(self) -> str: ...

class AccountBalance:
    def __init__(self, total: Money, locked: Money, free: Money): ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> AccountBalance: ...
    def to_dict(self) -> dict[str, str]: ...

class MarginBalance:
    def __init__(self, initial: Money, maintenance: Money, instrument_id: InstrumentId): ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> MarginBalance: ...
    def to_dict(self) -> dict[str, str]: ...

class AccountState:
    def __init__(
        self,
        account_id: AccountId,
        account_type: AccountType,
        base_currency: Currency | None,
        balances: list[AccountBalance],
        margins: list[MarginBalance],
        is_reported: bool,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> AccountState: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def account_type(self) -> AccountType: ...
    @property
    def base_currency(self) -> Currency | None: ...
    @property
    def balances(self) -> list[AccountBalance]: ...
    @property
    def margins(self) -> list[MarginBalance]: ...

# Instruments

class BettingInstrument:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        event_type_id: int,
        event_type_name: str,
        competition_id: int,
        competition_name: str,
        event_id: int,
        event_name: str,
        event_country_code: str,
        event_open_date: int,
        betting_type: str,
        market_id: str,
        market_name: str,
        market_type: str,
        market_start_time: int,
        selection_id: int,
        selection_name: str,
        selection_handicap: float,
        currency: Currency,
        price_precision: int,
        size_precision: int,
        price_increment: Price,
        size_increment: Quantity,
        ts_event: int,
        ts_init: int,
        outcome: str | None = None,
        description: str | None = None,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_notional: Money | None = None,
        min_notional: Money | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> BettingInstrument: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def asset_class(self) -> AssetClass: ...
    @property
    def instrument_class(self) -> InstrumentClass: ...
    @property
    def event_type_id(self) -> int: ...
    @property
    def event_type_name(self) -> str: ...
    @property
    def competition_id(self) -> int: ...
    @property
    def event_id(self) -> int: ...
    @property
    def event_name(self) -> str: ...
    @property
    def event_country_code(self) -> str: ...
    @property
    def event_open_date(self) -> str: ...
    @property
    def betting_type(self) -> str: ...
    @property
    def market_id(self) -> str: ...
    @property
    def market_name(self) -> str: ...
    @property
    def market_start_time(self) -> int: ...
    @property
    def selection_id(self) -> int: ...
    @property
    def selection_name(self) -> str: ...
    @property
    def currency(self) -> Currency: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_notional(self) -> Money | None: ...
    @property
    def min_notional(self) -> Money | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def info(self) -> dict[str, Any]: ...
    def to_dict(self) -> dict[str, Any]: ...

class BinaryOption:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        asset_class: AssetClass,
        currency: Currency,
        activation_ns: int,
        expiration_ns: int,
        price_precision: int,
        size_precision: int,
        price_increment: Price,
        size_increment: Quantity,
        ts_event: int,
        ts_init: int,
        outcome: str | None = None,
        description: str | None = None,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_notional: Money | None = None,
        min_notional: Money | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def asset_class(self) -> AssetClass: ...
    @property
    def currency(self) -> Currency: ...
    @property
    def activation_ns(self) -> int: ...
    @property
    def expiration_ns(self) -> int: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def outcome(self) -> str | None: ...
    @property
    def description(self) -> str | None: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_notional(self) -> Money | None: ...
    @property
    def min_notional(self) -> Money | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> BinaryOption: ...
    def to_dict(self) -> dict[str, Any]: ...

class CryptoFuture:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        underlying: Currency,
        quote_currency: Currency,
        settlement_currency: Currency,
        is_inverse: bool,
        activation_ns: int,
        expiration_ns: int,
        price_precision: int,
        size_precision: int,
        price_increment: Price,
        size_increment: Quantity,
        ts_event: int,
        ts_init: int,
        multiplier: Quantity | None = None,
        lot_size: Quantity | None = None,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_notional: Money | None = None,
        min_notional: Money | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def underlying(self) -> Currency: ...
    @property
    def quote_currency(self) -> Currency: ...
    @property
    def settlement_currency(self) -> Currency: ...
    @property
    def is_inverse(self) -> bool: ...
    @property
    def activation_ns(self) -> int: ...
    @property
    def expiration_ns(self) -> int: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def multiplier(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity | None: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_notional(self) -> Money | None: ...
    @property
    def min_notional(self) -> Money | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> CryptoFuture: ...
    def to_dict(self) -> dict[str, Any]: ...

class CryptoOption:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        underlying: Currency,
        quote_currency: Currency,
        settlement_currency: Currency,
        is_inverse: bool,
        option_kind: OptionKind,
        strike_price: Price,
        activation_ns: int,
        expiration_ns: int,
        price_precision: int,
        size_precision: int,
        price_increment: Price,
        size_increment: Quantity,
        ts_event: int,
        ts_init: int,
        multiplier: Quantity | None = None,
        lot_size: Quantity | None = None,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_notional: Money | None = None,
        min_notional: Money | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def underlying(self) -> Currency: ...
    @property
    def quote_currency(self) -> Currency: ...
    @property
    def settlement_currency(self) -> Currency: ...
    @property
    def is_inverse(self) -> bool: ...
    @property
    def option_kind(self) -> OptionKind: ...
    @property
    def strike_price(self) -> Price: ...
    @property
    def activation_ns(self) -> int: ...
    @property
    def expiration_ns(self) -> int: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def multiplier(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity | None: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_notional(self) -> Money | None: ...
    @property
    def min_notional(self) -> Money | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> CryptoOption: ...
    def to_dict(self) -> dict[str, Any]: ...

class CryptoPerpetual:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        base_currency: Currency,
        quote_currency: Currency,
        settlement_currency: Currency,
        is_inverse: bool,
        price_precision: int,
        size_precision: int,
        price_increment: Price,
        size_increment: Quantity,
        ts_event: int,
        ts_init: int,
        multiplier: Quantity | None = None,
        lot_size: Quantity | None = None,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_notional: Money | None = None,
        min_notional: Money | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def base_currency(self) -> Currency: ...
    @property
    def quote_currency(self) -> Currency: ...
    @property
    def settlement_currency(self) -> Currency: ...
    @property
    def is_inverse(self) -> bool: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def multiplier(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_notional(self) -> Money | None: ...
    @property
    def min_notional(self) -> Money | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> CryptoPerpetual: ...
    def to_dict(self) -> dict[str, Any]: ...

class CurrencyPair:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        base_currency: Currency,
        quote_currency: Currency,
        price_precision: int,
        size_precision: int,
        price_increment: Price,
        size_increment: Quantity,
        ts_event: int,
        ts_init: int,
        multiplier: Quantity | None = None,
        lot_size: Quantity | None = None,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def base_currency(self) -> Currency: ...
    @property
    def quote_currency(self) -> Currency: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity | None: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_notional(self) -> Money | None: ...
    @property
    def min_notional(self) -> Money | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> CurrencyPair: ...
    def to_dict(self) -> dict[str, Any]: ...

class Equity:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        currency: Currency,
        price_precision: int,
        price_increment: Price,
        ts_event: int,
        ts_init: int,
        isin: str | None = None,
        lot_size: Quantity | None = None,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def isin(self) -> str | None: ...
    @property
    def quote_currency(self) -> Currency: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity | None: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> Equity: ...
    def to_dict(self) -> dict[str, Any]: ...

class FuturesContract:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        asset_class: AssetClass,
        underlying: str,
        activation_ns: int,
        expiration_ns: int,
        currency: Currency,
        price_precision: int,
        price_increment: Price,
        multiplier: Quantity,
        lot_size: Quantity,
        ts_event: int,
        ts_init: int,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
        exchange: str | None = None,
        # info: dict[str, Any] | None = None,  # Add for pyo3
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def asset_class(self) -> AssetClass: ...
    @property
    def exchange(self) -> str: ...
    @property
    def underlying(self) -> str: ...
    @property
    def activation_ns(self) -> int: ...
    @property
    def expiration_ns(self) -> int: ...
    @property
    def currency(self) -> Currency: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def multiplier(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> CryptoFuture: ...
    def to_dict(self) -> dict[str, Any]: ...

class FuturesSpread:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        asset_class: AssetClass,
        underlying: str,
        strategy_type: str,
        activation_ns: int,
        expiration_ns: int,
        currency: Currency,
        price_precision: int,
        price_increment: Price,
        multiplier: Quantity,
        lot_size: Quantity,
        ts_event: int,
        ts_init: int,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
        exchange: str | None = None,
        # info: dict[str, Any] | None = None,  # Add for pyo3
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def asset_class(self) -> AssetClass: ...
    @property
    def exchange(self) -> str: ...
    @property
    def underlying(self) -> str: ...
    @property
    def strategy_type(self) -> str: ...
    @property
    def activation_ns(self) -> int: ...
    @property
    def expiration_ns(self) -> int: ...
    @property
    def currency(self) -> Currency: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def multiplier(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> FuturesSpread: ...
    def to_dict(self) -> dict[str, Any]: ...

class OptionContract:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        asset_class: AssetClass,
        underlying: str,
        option_kind: OptionKind,
        strike_price: Price,
        currency: Currency,
        activation_ns: int,
        expiration_ns: int,
        price_precision: int,
        price_increment: Price,
        multiplier: Quantity,
        lot_size: Quantity,
        ts_event: int,
        ts_init: int,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
        exchange: str | None = None,
        # info: dict[str, Any] | None = None,  # Add for pyo3
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def asset_class(self) -> AssetClass: ...
    @property
    def exchange(self) -> str: ...
    @property
    def underlying(self) -> str: ...
    @property
    def option_kind(self) -> OptionKind: ...
    @property
    def strike_price(self) -> Price: ...
    @property
    def activation_ns(self) -> int: ...
    @property
    def expiration_ns(self) -> int: ...
    @property
    def currency(self) -> Currency: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def multiplier(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OptionContract: ...
    def to_dict(self) -> dict[str, Any]: ...

class OptionSpread:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: Symbol,
        asset_class: AssetClass,
        underlying: str,
        strategy_type: str,
        activation_ns: int,
        expiration_ns: int,
        currency: Currency,
        price_precision: int,
        price_increment: Price,
        multiplier: Quantity,
        lot_size: Quantity,
        ts_event: int,
        ts_init: int,
        max_quantity: Quantity | None = None,
        min_quantity: Quantity | None = None,
        max_price: Price | None = None,
        min_price: Price | None = None,
        margin_init: Decimal | None = None,
        margin_maint: Decimal | None = None,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
        exchange: str | None = None,
        # info: dict[str, Any] | None = None,  # Add for pyo3
    ) -> None: ...
    @property
    def id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> Symbol: ...
    @property
    def asset_class(self) -> AssetClass: ...
    @property
    def exchange(self) -> str: ...
    @property
    def underlying(self) -> str: ...
    @property
    def strategy_type(self) -> str: ...
    @property
    def activation_ns(self) -> int: ...
    @property
    def expiration_ns(self) -> int: ...
    @property
    def currency(self) -> Currency: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    @property
    def price_increment(self) -> Price: ...
    @property
    def size_increment(self) -> Quantity: ...
    @property
    def multiplier(self) -> Quantity: ...
    @property
    def lot_size(self) -> Quantity: ...
    @property
    def max_quantity(self) -> Quantity | None: ...
    @property
    def min_quantity(self) -> Quantity | None: ...
    @property
    def max_price(self) -> Price | None: ...
    @property
    def min_price(self) -> Price | None: ...
    @property
    def margin_init(self) -> Decimal: ...
    @property
    def margin_maint(self) -> Decimal: ...
    @property
    def maker_fee(self) -> Decimal: ...
    @property
    def taker_fee(self) -> Decimal: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def info(self) -> dict[str, Any]: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OptionSpread: ...
    def to_dict(self) -> dict[str, Any]: ...

class SyntheticInstrument: ...

type Instrument = Union[
    BettingInstrument,
    BinaryOption,
    CryptoFuture,
    CryptoPerpetual,
    CurrencyPair,
    Equity,
    FuturesContract,
    FuturesSpread,
    OptionContract,
    OptionSpread,
]

# Events

class OrderDenied:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        reason: str,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderDenied: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def reason(self) -> str: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class OrderTriggered:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
        account_id: AccountId | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderRejected: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId | None: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderRejected:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        account_id: AccountId,
        reason: str,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        due_post_only: bool = False,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderRejected: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def reason(self) -> str: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...
    @property
    def due_post_only(self) -> bool: ...

class OrderFilled:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        venue_order_id: VenueOrderId,
        account_id: AccountId,
        trade_id: TradeId,
        order_side: OrderSide,
        order_type: OrderType,
        last_qty: Quantity,
        last_px: Price,
        currency: Currency,
        liquidity_side: LiquiditySide,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        position_id: PositionId | None = None,
        commission: Money | None = None,
    ) -> None: ...
    @property
    def is_buy(self) -> bool: ...
    @property
    def is_sell(self) -> bool: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderFilled: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def order_side(self) -> OrderSide: ...
    @property
    def order_type(self) -> OrderType: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...

class OrderInitialized:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        post_only: bool,
        reduce_only: bool,
        quote_quantity: bool,
        reconciliation: bool,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        price: Price | None = None,
        trigger_price: Price | None = None,
        trigger_type: TriggerType | None = None,
        limit_offset: Decimal | None = None,
        trailing_offset: Decimal | None = None,
        trailing_offset_type: TrailingOffsetType | None = None,
        expire_time: int | None = None,
        display_quantity: Quantity | None = None,
        emulation_trigger: TriggerType | None = None,
        trigger_instrument_id: InstrumentId | None = None,
        contingency_type: ContingencyType | None = None,
        order_list_id: OrderListId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        exec_algorithm_id: ExecAlgorithmId | None = None,
        exec_algorithm_params: dict[str, str] | None = None,
        exec_spawn_id: ClientOrderId | None = None,
        tags: list[str] | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderInitialized: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def order_type(self) -> OrderType: ...

class OrderSubmitted:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        account_id: AccountId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderSubmitted: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class OrderEmulated:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderEmulated: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class OrderReleased:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        released_price: Price,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderReleased: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def released_price(self) -> Price: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class OrderUpdated:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        quantity: Quantity,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
        account_id: AccountId | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderUpdated: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId | None: ...
    @property
    def quantity(self) -> Quantity: ...
    @property
    def price(self) -> Price | None: ...
    @property
    def trigger_price(self) -> Price | None: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderPendingUpdate:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        account_id: AccountId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderPendingUpdate: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderPendingCancel:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        account_id: AccountId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderPendingCancel: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderModifyRejected:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        reason: str,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
        account_id: AccountId | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderModifyRejected: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId | None: ...
    @property
    def reason(self) -> str: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderAccepted:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        venue_order_id: VenueOrderId,
        account_id: AccountId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderAccepted: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderCancelRejected:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        reason: str,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
        account_id: AccountId | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderCancelRejected: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId | None: ...
    @property
    def reason(self) -> str: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderCanceled:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
        account_id: AccountId | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderCanceled: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId | None: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderExpired:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
        reconciliation: bool,
        venue_order_id: VenueOrderId | None = None,
        account_id: AccountId | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderExpired: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def venue_order_id(self) -> VenueOrderId | None: ...
    @property
    def account_id(self) -> AccountId | None: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def reconciliation(self) -> bool: ...

class OrderSnapshot:
    @classmethod
    def from_dict(cls, values: dict[str, Any]) -> OrderSnapshot: ...

class PositionSnapshot:
    @classmethod
    def from_dict(cls, values: dict[str, Any]) -> PositionSnapshot: ...

class PositionAdjusted:
    def __init__(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        position_id: PositionId,
        account_id: AccountId,
        adjustment_type: PositionAdjustmentType,
        quantity_change: float | None,
        pnl_change: Money | None,
        reason: str | None,
        event_id: UUID4,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, Any]) -> PositionAdjusted: ...
    def to_dict(self) -> dict[str, Any]: ...
    @property
    def trader_id(self) -> TraderId: ...
    @property
    def strategy_id(self) -> StrategyId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def position_id(self) -> PositionId: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def adjustment_type(self) -> PositionAdjustmentType: ...
    @property
    def quantity_change(self) -> float | None: ...
    @property
    def pnl_change(self) -> Money | None: ...
    @property
    def reason(self) -> str | None: ...
    @property
    def event_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

# Execution reports

class FillReport:
    def __init__(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId,
        trade_id: TradeId,
        order_side: OrderSide,
        last_qty: Quantity,
        last_px: Price,
        commission: Money,
        liquidity_side: LiquiditySide,
        ts_event: int,
        ts_init: int,
        client_order_id: ClientOrderId | None = None,
        venue_position_id: PositionId | None = None,
        report_id: UUID4 | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> FillReport: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def venue_order_id(self) -> VenueOrderId: ...
    @property
    def client_order_id(self) -> ClientOrderId | None: ...
    @property
    def venue_position_id(self) -> PositionId | None: ...
    @property
    def trade_id(self) -> TradeId: ...
    @property
    def order_side(self) -> OrderSide: ...
    @property
    def last_qty(self) -> Quantity: ...
    @property
    def last_px(self) -> Price: ...
    @property
    def commission(self) -> Money: ...
    @property
    def liquidity_side(self) -> LiquiditySide: ...
    @property
    def report_id(self) -> UUID4: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class OrderStatusReport:
    def __init__(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        time_in_force: TimeInForce,
        order_status: OrderStatus,
        quantity: Quantity,
        filled_qty: Quantity,
        ts_accepted: int,
        ts_last: int,
        ts_init: int,
        client_order_id: ClientOrderId | None = None,
        order_list_id: OrderListId | None = None,
        venue_position_id: PositionId | None = None,
        linked_order_ids: list[ClientOrderId] | None = None,
        parent_order_id: ClientOrderId | None = None,
        contingency_type: ContingencyType | None = None,
        expire_time: int | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
        trigger_type: TriggerType | None = None,
        limit_offset: Decimal | None = None,
        trailing_offset: Decimal | None = None,
        trailing_offset_type: TrailingOffsetType | None = None,
        avg_px: Decimal | None = None,
        display_qty: Quantity | None = None,
        post_only: bool = False,
        reduce_only: bool = False,
        cancel_reason: str | None = None,
        ts_triggered: int | None = None,
        report_id: UUID4 | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> OrderStatusReport: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def venue_order_id(self) -> VenueOrderId: ...
    @property
    def order_side(self) -> OrderSide: ...
    @property
    def order_type(self) -> OrderType: ...
    @property
    def time_in_force(self) -> TimeInForce: ...
    @property
    def order_status(self) -> OrderStatus: ...
    @property
    def quantity(self) -> Quantity: ...
    @property
    def filled_qty(self) -> Quantity: ...
    @property
    def report_id(self) -> UUID4: ...
    @property
    def ts_accepted(self) -> int: ...
    @property
    def ts_last(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def client_order_id(self) -> ClientOrderId | None: ...
    @property
    def order_list_id(self) -> OrderListId | None: ...
    @property
    def venue_position_id(self) -> PositionId | None: ...
    @property
    def linked_order_ids(self) -> list[ClientOrderId] | None: ...
    @property
    def parent_order_id(self) -> ClientOrderId | None: ...
    @property
    def contingency_type(self) -> ContingencyType: ...
    @property
    def expire_time(self) -> int | None: ...
    @property
    def price(self) -> Price | None: ...
    @property
    def trigger_price(self) -> Price | None: ...
    @property
    def trigger_type(self) -> TriggerType | None: ...
    @property
    def limit_offset(self) -> Decimal | None: ...
    @property
    def trailing_offset(self) -> Decimal | None: ...
    @property
    def trailing_offset_type(self) -> TrailingOffsetType: ...
    @property
    def avg_px(self) -> Decimal | None: ...
    @property
    def display_qty(self) -> Quantity | None: ...
    @property
    def post_only(self) -> bool: ...
    @property
    def reduce_only(self) -> bool: ...
    @property
    def cancel_reason(self) -> str | None: ...
    @property
    def ts_triggered(self) -> int | None: ...
    @property
    def is_open(self) -> bool: ...

class PositionStatusReport:
    def __init__(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        position_side: PositionSide,
        quantity: Quantity,
        ts_last: int,
        ts_init: int,
        report_id: UUID4 | None = None,
        venue_position_id: PositionId | None = None,
        avg_px_open: Decimal | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> PositionStatusReport: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def venue_position_id(self) -> PositionId | None: ...
    @property
    def position_side(self) -> PositionSide: ...
    @property
    def quantity(self) -> Quantity: ...
    @property
    def avg_px_open(self) -> Decimal | None: ...
    @property
    def report_id(self) -> UUID4: ...
    @property
    def ts_last(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def is_flat(self) -> bool: ...
    @property
    def is_long(self) -> bool: ...
    @property
    def is_short(self) -> bool: ...

class ExecutionMassStatus:
    def __init__(
        self,
        client_id: ClientId,
        account_id: AccountId,
        venue: Venue,
        ts_init: int,
        report_id: UUID4 | None = None,
    ) -> None: ...
    @classmethod
    def from_dict(cls, values: dict[str, str]) -> ExecutionMassStatus: ...
    def to_dict(self) -> dict[str, str]: ...
    @property
    def client_id(self) -> ClientId: ...
    @property
    def account_id(self) -> AccountId: ...
    @property
    def venue(self) -> Venue: ...
    @property
    def report_id(self) -> UUID4: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def order_reports(self) -> dict[VenueOrderId, OrderStatusReport]: ...
    @property
    def fill_reports(self) -> dict[VenueOrderId, list[FillReport]]: ...
    @property
    def position_reports(self) -> dict[InstrumentId, list[PositionStatusReport]]: ...
    def add_order_reports(self, reports: list[OrderStatusReport]) -> None: ...
    def add_fill_reports(self, reports: list[FillReport]) -> None: ...
    def add_position_reports(self, reports: list[PositionStatusReport]) -> None: ...

# OrderBook

class BookLevel:
    @property
    def price(self) -> Price: ...
    def len(self) -> int: ...
    def is_empty(self) -> bool: ...
    def size(self) -> float: ...
    def size_raw(self) -> int: ...
    def exposure(self) -> float: ...
    def exposure_raw(self) -> int: ...
    def first(self) -> BookOrder | None: ...
    def get_orders(self) -> list[BookOrder]: ...

class OrderBook:
    def __init__(
        self,
        instrument_id: InstrumentId,
        book_type: BookType,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def book_type(self) -> BookType: ...
    @property
    def sequence(self) -> int: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    @property
    def ts_last(self) -> int: ...
    @property
    def update_count(self) -> int: ...
    def reset(self) -> None: ...
    def add(self, order: BookOrder, flags: int, sequence: int, ts_event: int) -> None: ...
    def update(self, order: BookOrder, flags: int, sequence: int, ts_event: int) -> None: ...
    def delete(self, order: BookOrder, flags: int, sequence: int, ts_event: int) -> None: ...
    def clear(self, sequence: int, ts_event: int) -> None: ...
    def clear_bids(self, sequence: int, ts_event: int) -> None: ...
    def clear_asks(self, sequence: int, ts_event: int) -> None: ...
    def clear_stale_levels(self, side: OrderSide | None = None) -> list[BookLevel] | None: ...
    def apply_delta(self, delta: OrderBookDelta) -> None: ...
    def apply_deltas(self, deltas: OrderBookDeltas) -> None: ...
    def apply_depth(self, depth: OrderBookDepth10) -> None: ...
    def bids(self, depth: int | None = None) -> list[BookLevel]: ...
    def asks(self, depth: int | None = None) -> list[BookLevel]: ...
    def bids_to_dict(self, depth: int | None = None) -> dict[Decimal, Decimal]: ...
    def asks_to_dict(self, depth: int | None = None) -> dict[Decimal, Decimal]: ...
    def group_bids(
        self, group_size: Decimal, depth: int | None = None
    ) -> dict[Decimal, Decimal]: ...
    def group_asks(
        self, group_size: Decimal, depth: int | None = None
    ) -> dict[Decimal, Decimal]: ...
    def bids_filtered_to_dict(
        self,
        depth: int | None = None,
        own_book: OwnOrderBook | None = None,
        status: set[OrderStatus] | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, Decimal]: ...
    def asks_filtered_to_dict(
        self,
        depth: int | None = None,
        own_book: OwnOrderBook | None = None,
        status: set[OrderStatus] | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, Decimal]: ...
    def group_bids_filtered(
        self,
        group_size: Decimal,
        depth: int | None = None,
        own_book: OwnOrderBook | None = None,
        status: set[OrderStatus] | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, Decimal]: ...
    def group_asks_filtered(
        self,
        group_size: Decimal,
        depth: int | None = None,
        own_book: OwnOrderBook | None = None,
        status: set[OrderStatus] | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, Decimal]: ...
    def best_bid_price(self) -> Price | None: ...
    def best_ask_price(self) -> Price | None: ...
    def best_bid_size(self) -> Quantity | None: ...
    def best_ask_size(self) -> Quantity | None: ...
    def spread(self) -> float | None: ...
    def midpoint(self) -> float | None: ...
    def get_avg_px_for_quantity(self, qty: Quantity, order_side: OrderSide) -> float: ...
    def get_quantity_for_price(self, price: Price, order_side: OrderSide) -> float: ...
    def get_quantity_at_level(
        self, price: Price, order_side: OrderSide, size_precision: int
    ) -> Quantity: ...
    def simulate_fills(self, order: BookOrder) -> list[tuple[Price, Quantity]]: ...
    def pprint(self, num_levels: int = 3, group_size: Decimal | None = None) -> str: ...

def update_book_with_quote_tick(book: OrderBook, quote: QuoteTick) -> None: ...
def update_book_with_trade_tick(book: OrderBook, trade: TradeTick) -> None: ...

class OwnBookOrder:
    def __init__(
        self,
        trader_id: TraderId,
        client_order_id: ClientOrderId,
        side: OrderSide,
        price: Price,
        size: Quantity,
        order_type: OrderType,
        time_in_force: TimeInForce,
        status: OrderStatus,
        ts_last: int,
        ts_accepted: int,
        ts_submitted: int,
        ts_init: int,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    @property
    def client_order_id(self) -> ClientOrderId: ...
    @property
    def side(self) -> OrderSide: ...
    @property
    def price(self) -> Price: ...
    @property
    def size(self) -> Quantity: ...
    @property
    def order_type(self) -> OrderType: ...
    @property
    def time_in_force(self) -> TimeInForce: ...
    @property
    def status(self) -> OrderStatus: ...
    @property
    def ts_last(self) -> int: ...
    @property
    def ts_init(self) -> int: ...
    def exposure(self) -> float: ...
    def signed_size(self) -> float: ...

class OwnOrderBook:
    def __init__(self, instrument_id: InstrumentId) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def ts_last(self) -> int: ...
    @property
    def update_count(self) -> int: ...
    def reset(self) -> None: ...
    def add(self, order: OwnBookOrder) -> None: ...
    def update(self, order: OwnBookOrder) -> None: ...
    def delete(self, order: OwnBookOrder) -> None: ...
    def clear(self) -> None: ...
    def bid_client_order_ids(self) -> list[ClientOrderId]: ...
    def ask_client_order_ids(self) -> list[ClientOrderId]: ...
    def is_order_in_book(self, client_order_id: ClientOrderId) -> bool: ...
    def orders_to_list(self) -> list[OwnBookOrder]: ...
    def bids_to_list(self) -> list[OwnBookOrder]: ...
    def asks_to_list(self) -> list[OwnBookOrder]: ...
    def bids_to_dict(
        self,
        status: set[OrderStatus] | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, list[OwnBookOrder]]: ...
    def asks_to_dict(
        self,
        status: set[OrderStatus] | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, list[OwnBookOrder]]: ...
    def bid_quantity(
        self,
        status: set[OrderStatus] | None = None,
        depth: int | None = None,
        group_size: Decimal | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, Decimal]: ...
    def ask_quantity(
        self,
        status: set[OrderStatus] | None = None,
        depth: int | None = None,
        group_size: Decimal | None = None,
        accepted_buffer_ns: int | None = None,
        ts_now: int | None = None,
    ) -> dict[Decimal, Decimal]: ...
    def pprint(self, num_levels: int = 3, group_size: Decimal | None = None) -> str: ...

###################################################################################################
# Infrastructure
###################################################################################################

class RedisMessageBusDatabase:
    def __init__(
        self,
        trader_id: TraderId,
        instance_id: UUID4,
        config_json: bytes,  # TODO: Standardize this back to `dict[str, Any]`
    ) -> None: ...
    def publish(self, topic: str, payload: bytes) -> None: ...
    def stream(self, callback: Callable) -> Awaitable[None]: ...
    def close(self) -> None: ...

class RedisCacheDatabase:
    def __init__(
        self,
        trader_id: TraderId,
        instance_id: UUID4,
        config: dict[str, Any],
    ) -> None: ...

class PostgresCacheDatabase:
    @classmethod
    def connect(
        cls,
        host: str | None = None,
        port: int | None = None,
        username: str | None = None,
        password: str | None = None,
        database: str | None = None,
    ) -> PostgresCacheDatabase: ...
    def close(self) -> None: ...
    def flush_db(self) -> None: ...
    def load(self) -> dict[str, str]: ...
    def load_currency(self, code: str) -> Currency | None: ...
    def load_currencies(self) -> list[Currency]: ...
    def load_instrument(self, instrument_id: InstrumentId) -> Instrument | None: ...
    def load_instruments(self) -> list[Instrument]: ...
    def load_order(self, client_order_id: ClientOrderId) -> Order | None: ...
    def load_account(self, account_id: AccountId) -> Account | None: ...
    def load_trades(self, instrument_id: InstrumentId) -> list[TradeTick]: ...
    def load_quotes(self, instrument_id: InstrumentId) -> list[QuoteTick]: ...
    def load_bars(self, instrument_id: InstrumentId) -> list[Bar]: ...
    def load_signals(self, name: str) -> list[Signal]: ...
    def load_custom_data(self, data_type: DataType) -> list[CustomData]: ...
    def load_order_snapshot(self, client_order_id: ClientOrderId) -> OrderSnapshot | None: ...
    def load_position_snapshot(self, position_id: PositionId) -> PositionSnapshot | None: ...
    def add(self, key: str, value: bytes) -> None: ...
    def add_currency(self, currency: Currency) -> None: ...
    def add_instrument(self, instrument: Instrument) -> None: ...
    def add_order(self, order: object) -> None: ...
    def add_order_snapshot(self, snapshot: OrderSnapshot) -> None: ...
    def add_position_snapshot(self, snapshot: PositionSnapshot) -> None: ...
    def add_account(self, account: object) -> None: ...
    def add_trade(self, trade: TradeTick) -> None: ...
    def add_quote(self, quote: QuoteTick) -> None: ...
    def add_bar(self, bar: Bar) -> None: ...
    def add_signal(self, signal: Signal) -> None: ...
    def add_custom_data(self, data: CustomData) -> None: ...
    def update_order(self, order: object) -> None: ...
    def update_account(self, account: Account) -> None: ...

class ParquetDataCatalogV2:
    def __init__(
        self,
        base_path: str,
        storage_options: dict[str, str] | None = None,
        batch_size: int | None = None,
        compression: int | None = None,
        max_row_group_size: int | None = None,
    ) -> None: ...
    def write_quote_ticks(
        self,
        data: list[QuoteTick],
        start: int | None = None,
        end: int | None = None,
        skip_disjoint_check: bool = False,
    ) -> str: ...
    def write_trade_ticks(
        self,
        data: list[TradeTick],
        start: int | None = None,
        end: int | None = None,
        skip_disjoint_check: bool = False,
    ) -> str: ...
    def write_order_book_deltas(
        self,
        data: list[OrderBookDelta],
        start: int | None = None,
        end: int | None = None,
        skip_disjoint_check: bool = False,
    ) -> str: ...
    def write_bars(
        self,
        data: list[Bar],
        start: int | None = None,
        end: int | None = None,
        skip_disjoint_check: bool = False,
    ) -> str: ...
    def write_order_book_depths(
        self,
        data: list[OrderBookDepth10],
        start: int | None = None,
        end: int | None = None,
        skip_disjoint_check: bool = False,
    ) -> str: ...
    def write_mark_price_updates(
        self,
        data: list[MarkPriceUpdate],
        start: int | None = None,
        end: int | None = None,
        skip_disjoint_check: bool = False,
    ) -> str: ...
    def write_index_price_updates(
        self,
        data: list[IndexPriceUpdate],
        start: int | None = None,
        end: int | None = None,
        skip_disjoint_check: bool = False,
    ) -> str: ...
    def consolidate_catalog(
        self,
        start: int | None = None,
        end: int | None = None,
        ensure_contiguous_files: bool | None = None,
    ) -> None: ...
    def consolidate_data(
        self,
        type_name: str,
        instrument_id: str | None = None,
        start: int | None = None,
        end: int | None = None,
        ensure_contiguous_files: bool | None = None,
    ) -> None: ...
    def consolidate_catalog_by_period(
        self,
        period_nanos: int | None = None,
        start: int | None = None,
        end: int | None = None,
        ensure_contiguous_files: bool | None = None,
    ) -> None: ...
    def consolidate_data_by_period(
        self,
        type_name: str,
        identifier: str | None = None,
        period_nanos: int | None = None,
        start: int | None = None,
        end: int | None = None,
        ensure_contiguous_files: bool | None = None,
    ) -> None: ...
    def query_last_timestamp(
        self,
        data_cls: str,
        instrument_id: str | None = None,
    ) -> int | None: ...
    def query_files(
        self,
        data_cls: str,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
    ) -> list[str]: ...
    def get_intervals(
        self,
        data_cls: str,
        instrument_id: str | None = None,
    ) -> list[tuple[int, int]]: ...
    def get_missing_intervals_for_request(
        self,
        start: int,
        end: int,
        data_cls: str,
        instrument_id: str | None = None,
    ) -> list[tuple[int, int]]: ...
    def reset_data_file_names(
        self,
        data_cls: str,
        instrument_id: str | None = None,
    ) -> None: ...
    def reset_all_file_names(self) -> None: ...
    def extend_file_name(
        self,
        data_cls: str,
        instrument_id: str | None = None,
        start: int | None = None,
        end: int | None = None,
    ) -> None: ...
    def query_quote_ticks(
        self,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
        where_clause: str | None = None,
    ) -> list[QuoteTick]: ...
    def query_trade_ticks(
        self,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
        where_clause: str | None = None,
    ) -> list[TradeTick]: ...
    def query_order_book_deltas(
        self,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
        where_clause: str | None = None,
    ) -> list[OrderBookDelta]: ...
    def query_bars(
        self,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
        where_clause: str | None = None,
    ) -> list[Bar]: ...
    def query_order_book_depths(
        self,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
        where_clause: str | None = None,
    ) -> list[OrderBookDepth10]: ...
    def query_mark_price_updates(
        self,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
        where_clause: str | None = None,
    ) -> list[MarkPriceUpdate]: ...
    def query_index_price_updates(
        self,
        instrument_ids: list[str] | None = None,
        start: int | None = None,
        end: int | None = None,
        where_clause: str | None = None,
    ) -> list[IndexPriceUpdate]: ...
    def delete_data_range(
        self,
        type_name: str,
        identifier: str | None = None,
        start: int | None = None,
        end: int | None = None,
    ) -> None: ...
    def delete_catalog_range(
        self,
        start: int | None = None,
        end: int | None = None,
    ) -> None: ...

###################################################################################################
# Network
###################################################################################################

class HttpError(Exception): ...
class HttpTimeoutError(Exception): ...

class HttpClient:
    def __init__(
        self,
        default_headers: dict[str, str] | None = None,
        header_keys: list[str] | None = None,
        keyed_quotas: list[tuple[str, Quota]] | None = None,
        default_quota: Quota | None = None,
        timeout_secs: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    async def request(
        self,
        method: HttpMethod,
        url: str,
        params: dict[str, Any] | None = None,
        headers: dict[str, str] | None = None,
        body: bytes | None = None,
        keys: list[str] | None = None,
        timeout_secs: int | None = None,
    ) -> HttpResponse: ...
    async def get(
        self,
        url: str,
        params: dict[str, Any] | None = None,
        headers: dict[str, str] | None = None,
        keys: list[str] | None = None,
        timeout_secs: int | None = None,
    ) -> HttpResponse: ...
    async def post(
        self,
        url: str,
        params: dict[str, Any] | None = None,
        headers: dict[str, str] | None = None,
        body: bytes | None = None,
        keys: list[str] | None = None,
        timeout_secs: int | None = None,
    ) -> HttpResponse: ...
    async def patch(
        self,
        url: str,
        params: dict[str, Any] | None = None,
        headers: dict[str, str] | None = None,
        body: bytes | None = None,
        keys: list[str] | None = None,
        timeout_secs: int | None = None,
    ) -> HttpResponse: ...
    async def delete(
        self,
        url: str,
        params: dict[str, Any] | None = None,
        headers: dict[str, str] | None = None,
        keys: list[str] | None = None,
        timeout_secs: int | None = None,
    ) -> HttpResponse: ...

class HttpMethod(Enum):
    GET = "GET"
    POST = "POST"
    PUT = "PUT"
    DELETE = "DELETE"
    PATCH = "PATCH"

class HttpResponse:
    @property
    def status(self) -> int: ...
    @property
    def body(self) -> bytes: ...
    @property
    def headers(self) -> dict[str, str]: ...

def http_get(
    url: str,
    params: dict[str, Any] | None = None,
    headers: dict[str, str] | None = None,
    timeout_secs: int | None = None,
) -> HttpResponse: ...
def http_post(
    url: str,
    params: dict[str, Any] | None = None,
    headers: dict[str, str] | None = None,
    body: bytes | None = None,
    timeout_secs: int | None = None,
) -> HttpResponse: ...
def http_patch(
    url: str,
    params: dict[str, Any] | None = None,
    headers: dict[str, str] | None = None,
    body: bytes | None = None,
    timeout_secs: int | None = None,
) -> HttpResponse: ...
def http_delete(
    url: str,
    params: dict[str, Any] | None = None,
    headers: dict[str, str] | None = None,
    timeout_secs: int | None = None,
) -> HttpResponse: ...
def http_download(
    url: str,
    filepath: str,
    params: dict[str, Any] | None = None,
    headers: dict[str, str] | None = None,
    timeout_secs: int | None = None,
) -> None: ...

class Quota:
    @classmethod
    def rate_per_second(cls, max_burst: int) -> Quota: ...
    @classmethod
    def rate_per_minute(cls, max_burst: int) -> Quota: ...
    @classmethod
    def rate_per_hour(cls, max_burst: int) -> Quota: ...

class WebSocketClientError(Exception): ...

class WebSocketConfig:
    def __init__(
        self,
        url: str,
        headers: list[tuple[str, str]],
        heartbeat: int | None = None,
        heartbeat_msg: str | None = None,
        reconnect_timeout_ms: int | None = 10_000,
        reconnect_delay_initial_ms: int | None = 2_000,
        reconnect_delay_max_ms: int | None = 30_000,
        reconnect_backoff_factor: float | None = 1.5,
        reconnect_jitter_ms: int | None = 100,
        reconnect_max_attempts: int | None = None,
    ) -> None: ...

class WebSocketClient:
    @classmethod
    def connect(
        cls,
        loop_: Any,
        config: WebSocketConfig,
        handler: Callable[..., Any],
        ping_handler: Callable[..., Any] | None = None,
        post_reconnection: Callable[..., None] | None = None,
        keyed_quotas: list[tuple[str, Quota]] = [],
        default_quota: Quota | None = None,
    ) -> Awaitable[WebSocketClient]: ...
    def disconnect(self) -> Awaitable[None]: ...
    def is_active(self) -> bool: ...
    def is_reconnecting(self) -> bool: ...
    def is_disconnecting(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def send(self, data: bytes, keys: list[str] | None = None) -> Awaitable[None]: ...
    def send_text(self, data: bytes, keys: list[str] | None = None) -> Awaitable[None]: ...
    def send_pong(self, data: bytes) -> Awaitable[None]: ...

class SocketConfig:
    def __init__(
        self,
        url: str,
        ssl: bool,
        suffix: bytes,
        handler: Callable[..., Any],
        heartbeat: tuple[int, bytes] | None = None,
        reconnect_timeout_ms: int | None = 10_000,
        reconnect_delay_initial_ms: int | None = 2_000,
        reconnect_delay_max_ms: int | None = 30_000,
        reconnect_backoff_factor: float | None = 1.5,
        reconnect_jitter_ms: int | None = 100,
        certs_dir: str | None = None,
    ) -> None: ...

class SocketClient:
    @classmethod
    def connect(
        cls,
        config: SocketConfig,
        post_connection: Callable[..., None] | None = None,
        post_reconnection: Callable[..., None] | None = None,
        post_disconnection: Callable[..., None] | None = None,
    ) -> Awaitable[SocketClient]: ...
    def is_active(self) -> bool: ...
    def is_reconnecting(self) -> bool: ...
    def is_disconnecting(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def mode(self) -> str: ...
    def reconnect(self) -> Awaitable[None]: ...
    def close(self) -> Awaitable[None]: ...
    def send(self, data: bytes) -> Awaitable[None]: ...

###################################################################################################
# Persistence
###################################################################################################

class NautilusDataType(Enum):
    OrderBookDelta = 1
    OrderBookDepth10 = 2
    QuoteTick = 3
    TradeTick = 4
    Bar = 5
    MarkPriceUpdate = 6

class DataBackendSession:
    def __init__(self, chunk_size: int = 10_000) -> None: ...
    def add_file(
        self,
        data_type: NautilusDataType,
        table_name: str,
        file_path: str,
        sql_query: str | None = None,
    ) -> None: ...
    def register_object_store_from_uri(
        self, uri: str, storage_options: dict[str, str] | None = None
    ) -> None: ...
    def to_query_result(self) -> DataQueryResult: ...

class QueryResult:
    def next(self) -> Data | None: ...

class DataQueryResult:
    def __init__(self, result: QueryResult, size: int) -> None: ...
    def __iter__(self) -> DataQueryResult: ...
    def __next__(self) -> Any | None: ...

class OrderBookDeltaDataWrangler:
    def __init__(
        self,
        instrument_id: str,
        price_precision: int,
        size_precision: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> str: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    def process_record_batch_bytes(self, data: bytes) -> list[OrderBookDelta]: ...

class OrderBookDepth10DataWrangler:
    def __init__(
        self,
        instrument_id: str,
        price_precision: int,
        size_precision: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> str: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    def process_record_batch_bytes(self, data: bytes) -> list[OrderBookDepth10]: ...

class QuoteTickDataWrangler:
    def __init__(
        self,
        instrument_id: str,
        price_precision: int,
        size_precision: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> str: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    def process_record_batch_bytes(self, data: bytes) -> list[QuoteTick]: ...

class TradeTickDataWrangler:
    def __init__(
        self,
        instrument_id: str,
        price_precision: int,
        size_precision: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> str: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    def process_record_batch_bytes(self, data: bytes) -> list[TradeTick]: ...

class BarDataWrangler:
    def __init__(
        self,
        bar_type: str,
        price_precision: int,
        size_precision: int,
    ) -> None: ...
    @property
    def bar_type(self) -> str: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...
    def process_record_batch_bytes(self, data: bytes) -> list[Bar]: ...

###################################################################################################
# Serialization
###################################################################################################

def get_arrow_schema_map(data_cls: type) -> dict[str, str]: ...
def pyobjects_to_arrow_record_batch_bytes(data: list[Data]) -> bytes: ...
def book_deltas_to_arrow_record_batch_bytes(data: list[OrderBookDelta]) -> bytes: ...
def book_depth10_to_arrow_record_batch_bytes(data: list[OrderBookDepth10]) -> bytes: ...
def quotes_to_arrow_record_batch_bytes(data: list[QuoteTick]) -> bytes: ...
def trades_to_arrow_record_batch_bytes(data: list[TradeTick]) -> bytes: ...
def bars_to_arrow_record_batch_bytes(data: list[Bar]) -> bytes: ...
def mark_prices_to_arrow_record_batch_bytes(data: list[IndexPriceUpdate]) -> bytes: ...
def index_prices_to_arrow_record_batch_bytes(data: list[IndexPriceUpdate]) -> bytes: ...
def instrument_closes_to_arrow_record_batch_bytes(data: list[InstrumentClose]) -> bytes: ...

###################################################################################################
# Indicators
###################################################################################################

class AdaptiveMovingAverage:
    def __init__(
        self,
        period_efficiency_ratio: int,
        period_fast: int,
        period_slow: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    def update_raw(self, value: float) -> None: ...
    def reset(self) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...

class SimpleMovingAverage:
    def __init__(
        self,
        period: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, value: float) -> None: ...
    def reset(self) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...

class ExponentialMovingAverage:
    def __init__(
        self,
        period: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    @property
    def alpha(self) -> float: ...
    def update_raw(self, value: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class LinearRegression:
    def __init__(
        self,
        period: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def slope(self) -> float: ...
    @property
    def intercept(self) -> float: ...
    @property
    def degree(self) -> float: ...
    @property
    def cfo(self) -> float: ...
    @property
    def r2(self) -> float: ...
    @property
    def value(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    def update_raw(self, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class DoubleExponentialMovingAverage:
    def __init__(
        self,
        period: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, value: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class HullMovingAverage:
    def __init__(
        self,
        period: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, value: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class WilderMovingAverage:
    def __init__(
        self,
        period: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    @property
    def alpha(self) -> float: ...
    def update_raw(self, value: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class VariableIndexDynamicAverage:
    def __init__(
        self,
        period: int,
        cmo_ma_type: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    @property
    def alpha(self) -> float: ...
    @property
    def cmo_pct(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class VolumeWeightedAveragePrice:
    def __init__(
        self,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, price: float, volume: float, timestamp: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class PsychologicalLine:
    def __init__(
        self,
        period: int,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class VerticalHorizontalFilter:
    def __init__(
        self,
        period: int,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class ChandeMomentumOscillator:
    def __init__(
        self,
        period: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class KlingerVolumeOscillator:
    def __init__(
        self,
        fast_period: int,
        slow_period: int,
        signal_period: int,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def fast_period(self) -> int: ...
    @property
    def slow_period(self) -> int: ...
    @property
    def signal_period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float, volume: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class DirectionalMovement:
    def __init__(
        self,
        period: int,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def pos(self) -> float: ...
    @property
    def neg(self) -> float: ...
    def update_raw(self, high: float, low: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class ArcherMovingAveragesTrends:
    def __init__(
        self,
        fast_period: int,
        slow_period: int,
        signal_period: int,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def fast_period(self) -> int: ...
    @property
    def slow_period(self) -> int: ...
    @property
    def signal_period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def long_run(self) -> bool: ...
    @property
    def short_run(self) -> bool: ...
    def update_raw(self, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class Swings:
    def __init__(
        self,
        period: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def direction(self) -> float: ...
    @property
    def changed(self) -> bool: ...
    @property
    def high_datetime(self) -> float: ...
    @property
    def low_datetime(self) -> float: ...
    @property
    def high_price(self) -> float: ...
    @property
    def low_price(self) -> float: ...
    @property
    def duration(self) -> int: ...
    @property
    def since_high(self) -> int: ...
    @property
    def since_low(self) -> int: ...
    @property
    def length(self) -> int: ...
    def update_raw(self, high: float, low: float, timestamp: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class BollingerBands:
    def __init__(
        self,
        period: int,
        k: float,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def k(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def upper(self) -> float: ...
    @property
    def middle(self) -> float: ...
    @property
    def lower(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class Stochastics:
    def __init__(
        self,
        period_k: int,
        period_d: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period_k(self) -> int: ...
    @property
    def period_d(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value_k(self) -> float: ...
    @property
    def value_d(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class VolatilityRatio:
    def __init__(
        self,
        fast_period: int,
        slow_period: int,
        use_previous: bool,
        value_floor: float,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def fast_period(self) -> int: ...
    @property
    def slow_period(self) -> int: ...
    @property
    def use_previous(self) -> bool: ...
    @property
    def value_floor(self) -> float: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class Pressure:
    def __init__(
        self,
        period: int,
        ma_type: MovingAverageType = ...,
        atr_floor: float = 0.0,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    @property
    def value_cumulative(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float, volume: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class AroonOscillator:
    def __init__(
        self,
        period: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    @property
    def aroon_up(self) -> float: ...
    @property
    def aroon_down(self) -> float: ...
    def update_raw(self, high: float, low: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class Bias:
    def __init__(
        self,
        period: int,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class EfficiencyRatio:
    def __init__(
        self,
        period: int,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def value(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    def update_raw(self, close: float) -> None: ...

class RelativeStrengthIndex:
    def __init__(
        self,
        period: int,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...

class RelativeVolatilityIndex:
    def __init__(
        self,
        period: int,
        scalar: float,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def scalar(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class RateOfChange:
    def __init__(
        self,
        period: int,
        use_log: bool,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def use_log(self) -> bool: ...
    @property
    def value(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    def update_raw(self, price: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class MovingAverageConvergenceDivergence:
    def __init__(
        self,
        fast_period: int,
        slow_period: int,
        ma_type: MovingAverageType = ...,
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def count(self) -> int: ...
    @property
    def fast_period(self) -> int: ...
    @property
    def slow_period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, close: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class OnBalanceVolume:
    def __init__(
        self,
        period: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, open: float, close: float, volume: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class AverageTrueRange:
    def __init__(
        self,
        period: int,
        ma_type: MovingAverageType = ...,
        use_previous: bool = True,
        value_floor: float = 0.0,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class CommodityChannelIndex:
    def __init__(
        self,
        period: int,
        scalar: float,
        ma_type: MovingAverageType = ...,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def scalar(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class DonchianChannel:
    def __init__(
        self,
        period: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def upper(self) -> float: ...
    @property
    def middle(self) -> float: ...
    @property
    def lower(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    def update_raw(self, high: float, low: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class KeltnerChannel:
    def __init__(
        self,
        period: int,
        k_multiplier: float,
        ma_type: MovingAverageType = ...,
        ma_type_atr: MovingAverageType = ...,
        use_previous: bool = True,
        atr_floor: float = 0.0,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def k_multiplier(self) -> float: ...
    @property
    def use_previous(self) -> bool: ...
    @property
    def atr_floor(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def upper(self) -> float: ...
    @property
    def middle(self) -> float: ...
    @property
    def lower(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class FuzzyCandle:
    def __init__(
        self,
        direction: CandleDirection,
        size: CandleSize,
        body_size: CandleBodySize,
        upper_wick_size: CandleWickSize,
        lower_wick_size: CandleWickSize,
    ) -> None: ...
    @property
    def direction(self) -> CandleDirection: ...
    @property
    def size(self) -> CandleBodySize: ...
    @property
    def body_size(self) -> CandleBodySize: ...
    @property
    def upper_wick_size(self) -> CandleWickSize: ...
    @property
    def lower_wick_size(self) -> CandleWickSize: ...

class FuzzyCandlesticks:
    def __init__(
        self,
        period: int,
        threshold1: float,
        threshold2: float,
        threshold3: float,
        threshold4: float,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def threshold1(self) -> float: ...
    @property
    def threshold2(self) -> float: ...
    @property
    def threshold3(self) -> float: ...
    @property
    def threshold4(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> FuzzyCandle: ...
    @property
    def vector(self) -> list[int]: ...
    def update_raw(self, open: float, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

# Fuzzy Enums
class CandleBodySize(Enum):
    NONE = 0
    SMALL = 1
    MEDIUM = 2
    LARGE = 3
    TREND = 4

class CandleDirection(Enum):
    BULL = 1
    NONE = 0
    BEAR = -1

class CandleSize(Enum):
    NONE = 0
    VERY_SMALL = 1
    SMALL = 2
    MEDIUM = 3
    LARGE = 4
    VERY_LARGE = 5
    EXTREMELY_LARGE = 6

class CandleWickSize(Enum):
    NONE = 0
    SMALL = 1
    MEDIUM = 2
    LARGE = 3

class SpreadAnalyzer:
    def __init__(
        self,
        instrument_id: InstrumentId,
        capacity: int,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def capacity(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def current(self) -> float: ...
    @property
    def average(self) -> float: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def reset(self) -> None: ...

class KeltnerPosition:
    def __init__(
        self,
        period: int,
        k_multiplier: float,
        ma_type: MovingAverageType = ...,
        ma_type_atr: MovingAverageType = ...,
        use_previous: bool = True,
        atr_floor: float = 0.0,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def k_multiplier(self) -> float: ...
    @property
    def use_previous(self) -> bool: ...
    @property
    def atr_floor(self) -> float: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def update_raw(self, high: float, low: float, close: float) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

class WeightedMovingAverage:
    def __init__(
        self,
        period: int,
        weights: list[float],
        price_type: PriceType | None = None,
    ) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def period(self) -> int: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    def update_raw(self, value: float) -> None: ...
    def handle_quote_tick(self, quote: QuoteTick) -> None: ...
    def handle_trade_tick(self, trade: TradeTick) -> None: ...
    def handle_bar(self, bar: Bar) -> None: ...
    def reset(self) -> None: ...

# Book

class BookImbalanceRatio:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    @property
    def count(self) -> int: ...
    @property
    def initialized(self) -> bool: ...
    @property
    def has_inputs(self) -> bool: ...
    @property
    def value(self) -> float: ...
    def handle_book(self, book: OrderBook) -> None: ...
    def update(self, best_bid: Quantity | None, best_ask: Quantity) -> None: ...
    def reset(self) -> None: ...

###################################################################################################
# Adapters
###################################################################################################

# AX Exchange

class AxEnvironment(Enum):
    SANDBOX = "SANDBOX"
    PRODUCTION = "PRODUCTION"

    @classmethod
    def from_str(cls, value: str) -> AxEnvironment: ...

class AxMarketDataLevel(Enum):
    LEVEL_1 = "LEVEL_1"
    LEVEL_2 = "LEVEL_2"
    LEVEL_3 = "LEVEL_3"

    @classmethod
    def from_str(cls, value: str) -> AxMarketDataLevel: ...

class AxHttpClient:
    def __init__(
        self,
        base_url: str | None = None,
        orders_base_url: str | None = None,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @staticmethod
    def with_credentials(
        api_key: str,
        api_secret: str,
        base_url: str | None = None,
        orders_base_url: str | None = None,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        proxy_url: str | None = None,
    ) -> AxHttpClient: ...
    @property
    def base_url(self) -> str: ...
    def cancel_all_requests(self) -> None: ...
    def cache_instrument(self, instrument: Any) -> None: ...
    async def authenticate(
        self,
        api_key: str,
        api_secret: str,
        expiration_seconds: int = 86400,
    ) -> str: ...
    async def authenticate_auto(self, expiration_seconds: int = 86400) -> str: ...
    async def request_instruments(
        self,
        maker_fee: Decimal | None = None,
        taker_fee: Decimal | None = None,
    ) -> list[Any]: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
    ) -> list[FillReport]: ...
    async def request_position_reports(
        self,
        account_id: AccountId,
    ) -> list[PositionStatusReport]: ...

class AxMdWebSocketClient:
    def __init__(
        self,
        url: str,
        auth_token: str,
        heartbeat: int | None = None,
    ) -> None: ...
    @staticmethod
    def without_auth(url: str, heartbeat: int | None = None) -> AxMdWebSocketClient: ...
    @property
    def url(self) -> str: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def subscription_count(self) -> int: ...
    def set_auth_token(self, token: str) -> None: ...
    def cache_instrument(self, instrument: Any) -> None: ...
    async def connect(self, callback: Callable[[Any], None]) -> None: ...
    async def subscribe(self, symbol: str, level: AxMarketDataLevel) -> None: ...
    async def unsubscribe(self, symbol: str) -> None: ...
    async def disconnect(self) -> None: ...
    async def close(self) -> None: ...

# Bybit

BYBIT_NAUTILUS_BROKER_ID: Final[str]

class BybitAccountType(Enum):
    UNIFIED = "UNIFIED"

class BybitProductType(Enum):
    SPOT = "spot"
    LINEAR = "linear"
    INVERSE = "inverse"
    OPTION = "option"

class BybitEnvironment(Enum):
    MAINNET = "mainnet"
    DEMO = "demo"
    TESTNET = "testnet"

class BybitMarginMode(Enum):
    ISOLATED_MARGIN = "ISOLATED_MARGIN"
    REGULAR_MARGIN = "REGULAR_MARGIN"
    PORTFOLIO_MARGIN = "PORTFOLIO_MARGIN"

class BybitMarginAction(Enum):
    BORROW = "borrow"
    REPAY = "repay"
    GET_BORROW_AMOUNT = "get_borrow_amount"

class BybitOpenOnly(Enum):
    OpenOnly = 0
    ClosedRecent = 1

class BybitOrderFilter(Enum):
    Order = "Order"
    StopOrder = "StopOrder"
    TpslOrder = "tpslOrder"
    OcoOrder = "OcoOrder"
    BidirectionalTpslOrder = "BidirectionalTpslOrder"

class BybitOrderSide(Enum):
    Unknown = "Unknown"
    Buy = "Buy"
    Sell = "Sell"

class BybitOrderStatus(Enum):
    Created = "Created"
    New = "New"
    Rejected = "Rejected"
    PartiallyFilled = "PartiallyFilled"
    PartiallyFilledCanceled = "PartiallyFilledCanceled"
    Filled = "Filled"
    Canceled = "Canceled"
    Untriggered = "Untriggered"
    Triggered = "Triggered"
    Deactivated = "Deactivated"

class BybitOrderType(Enum):
    Market = "Market"
    Limit = "Limit"
    Unknown = "Unknown"

class BybitCancelType(Enum):
    CancelByUser = "CancelByUser"
    CancelByReduceOnly = "CancelByReduceOnly"
    CancelByPrepareLackOfMargin = "CancelByPrepareLackOfMargin"
    CancelByPrepareOrderFilter = "CancelByPrepareOrderFilter"
    CancelByPrepareOrderMarginCheckFailed = "CancelByPrepareOrderMarginCheckFailed"
    CancelByPrepareOrderCommission = "CancelByPrepareOrderCommission"
    CancelByPrepareOrderRms = "CancelByPrepareOrderRms"
    CancelByPrepareOrderOther = "CancelByPrepareOrderOther"
    CancelByRiskLimit = "CancelByRiskLimit"
    CancelOnDisconnect = "CancelOnDisconnect"
    CancelByStopOrdersExceeded = "CancelByStopOrdersExceeded"
    CancelByPzMarketClose = "CancelByPzMarketClose"
    CancelByMarginCheckFailed = "CancelByMarginCheckFailed"
    CancelByPzTakeover = "CancelByPzTakeover"
    CancelByAdmin = "CancelByAdmin"
    CancelByTpSlTsClear = "CancelByTpSlTsClear"
    CancelByAmendNotModified = "CancelByAmendNotModified"
    CancelByPzCancel = "CancelByPzCancel"
    CancelByCrossSelfMatch = "CancelByCrossSelfMatch"
    CancelBySelfMatchPrevention = "CancelBySelfMatchPrevention"
    Other = "Other"

class BybitStopOrderType(Enum):
    NONE = "None"
    Unknown = "Unknown"
    TakeProfit = "TakeProfit"
    StopLoss = "StopLoss"
    TrailingStop = "TrailingStop"
    Stop = "Stop"
    PartialTakeProfit = "PartialTakeProfit"
    PartialStopLoss = "PartialStopLoss"
    TpslOrder = "TpslOrder"
    OcoOrder = "OcoOrder"
    MmRateClose = "MmRateClose"
    BidirectionalTpslOrder = "BidirectionalTpslOrder"

class BybitTriggerType(Enum):
    NONE = "None"
    LastPrice = "LastPrice"
    IndexPrice = "IndexPrice"
    MarkPrice = "MarkPrice"

class BybitTriggerDirection(Enum):
    NONE = 0
    RisesTo = 1
    FallsTo = 2

class BybitTpSlMode(Enum):
    Full = "Full"
    Partial = "Partial"
    Unknown = "Unknown"

class BybitTimeInForce(Enum):
    GTC = "GTC"
    IOC = "IOC"
    FOK = "FOK"
    PostOnly = "PostOnly"

class BybitMarginBorrowResult:
    def __init__(
        self,
        coin: str,
        amount: str,
        success: bool,
        message: str,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def coin(self) -> str: ...
    @property
    def amount(self) -> str: ...
    @property
    def success(self) -> bool: ...
    @property
    def message(self) -> str: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class BybitMarginRepayResult:
    def __init__(
        self,
        coin: str,
        amount: str | None,
        success: bool,
        result_status: str,
        message: str,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def coin(self) -> str: ...
    @property
    def amount(self) -> str | None: ...
    @property
    def success(self) -> bool: ...
    @property
    def result_status(self) -> str: ...
    @property
    def message(self) -> str: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class BybitMarginStatusResult:
    def __init__(
        self,
        coin: str,
        borrow_amount: str,
        ts_event: int,
        ts_init: int,
    ) -> None: ...
    @property
    def coin(self) -> str: ...
    @property
    def borrow_amount(self) -> str: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class BybitPositionMode(Enum):
    MergedSingle = 0
    BothSides = 3

class BybitServerTime:
    @property
    def time_second(self) -> str: ...
    @property
    def time_nano(self) -> str: ...

class BybitOrder:
    @property
    def order_id(self) -> str: ...
    @property
    def order_link_id(self) -> str: ...
    @property
    def block_trade_id(self) -> str | None: ...
    @property
    def symbol(self) -> str: ...
    @property
    def price(self) -> str: ...
    @property
    def qty(self) -> str: ...
    @property
    def side(self) -> BybitOrderSide: ...
    @property
    def is_leverage(self) -> str: ...
    @property
    def position_idx(self) -> int: ...
    @property
    def order_status(self) -> BybitOrderStatus: ...
    @property
    def cancel_type(self) -> BybitCancelType: ...
    @property
    def reject_reason(self) -> str: ...
    @property
    def avg_price(self) -> str | None: ...
    @property
    def leaves_qty(self) -> str: ...
    @property
    def leaves_value(self) -> str: ...
    @property
    def cum_exec_qty(self) -> str: ...
    @property
    def cum_exec_value(self) -> str: ...
    @property
    def cum_exec_fee(self) -> str: ...
    @property
    def time_in_force(self) -> BybitTimeInForce: ...
    @property
    def order_type(self) -> BybitOrderType: ...
    @property
    def stop_order_type(self) -> BybitStopOrderType: ...
    @property
    def order_iv(self) -> str | None: ...
    @property
    def trigger_price(self) -> str: ...
    @property
    def take_profit(self) -> str: ...
    @property
    def stop_loss(self) -> str: ...
    @property
    def tp_trigger_by(self) -> BybitTriggerType: ...
    @property
    def sl_trigger_by(self) -> BybitTriggerType: ...
    @property
    def trigger_direction(self) -> BybitTriggerDirection: ...
    @property
    def trigger_by(self) -> BybitTriggerType: ...
    @property
    def last_price_on_created(self) -> str: ...
    @property
    def reduce_only(self) -> bool: ...
    @property
    def close_on_trigger(self) -> bool: ...
    @property
    def smp_type(self) -> str: ...
    @property
    def smp_group(self) -> int: ...
    @property
    def smp_order_id(self) -> str: ...
    @property
    def tpsl_mode(self) -> BybitTpSlMode | None: ...
    @property
    def tp_limit_price(self) -> str: ...
    @property
    def sl_limit_price(self) -> str: ...
    @property
    def place_type(self) -> str: ...
    @property
    def created_time(self) -> str: ...
    @property
    def updated_time(self) -> str: ...

class BybitOrderCursorList:
    @property
    def list(self) -> list[BybitOrder]: ...
    @property
    def next_page_cursor(self) -> str | None: ...
    @property
    def category(self) -> BybitProductType | None: ...

class BybitTickerData:
    @property
    def symbol(self) -> str: ...
    @property
    def bid1_price(self) -> str: ...
    @property
    def bid1_size(self) -> str: ...
    @property
    def ask1_price(self) -> str: ...
    @property
    def ask1_size(self) -> str: ...
    @property
    def last_price(self) -> str: ...
    @property
    def high_price24h(self) -> str: ...
    @property
    def low_price24h(self) -> str: ...
    @property
    def turnover24h(self) -> str: ...
    @property
    def volume24h(self) -> str: ...

class BybitWebSocketError:
    @property
    def code(self) -> int: ...
    @property
    def message(self) -> str: ...
    @property
    def conn_id(self) -> str | None: ...
    @property
    def topic(self) -> str | None: ...
    @property
    def req_id(self) -> str | None: ...

class BybitAccountDetails:
    @property
    def id(self) -> str: ...
    @property
    def note(self) -> str: ...
    @property
    def api_key(self) -> str: ...
    @property
    def read_only(self) -> int: ...
    @property
    def key_type(self) -> int: ...
    @property
    def user_id(self) -> int | None: ...
    @property
    def inviter_id(self) -> int | None: ...
    @property
    def vip_level(self) -> str: ...
    @property
    def mkt_maker_level(self) -> int: ...
    @property
    def affiliate_id(self) -> int | None: ...
    @property
    def rsa_public_key(self) -> str: ...
    @property
    def is_master(self) -> bool: ...
    @property
    def parent_uid(self) -> str: ...
    @property
    def uta(self) -> int: ...
    @property
    def kyc_level(self) -> str: ...
    @property
    def kyc_region(self) -> str: ...
    @property
    def deadline_day(self) -> int: ...
    @property
    def expired_at(self) -> str | None: ...
    @property
    def created_at(self) -> str: ...

class BybitRawHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        demo: bool = False,
        testnet: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        recv_window_ms: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @property
    def base_url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def recv_window_ms(self) -> int: ...
    def cancel_all_requests(self) -> None: ...
    async def get_server_time(self) -> BybitServerTime: ...
    async def get_open_orders(
        self,
        category: BybitProductType,
        symbol: str | None = None,
        base_coin: str | None = None,
        settle_coin: str | None = None,
        order_id: str | None = None,
        order_link_id: str | None = None,
        open_only: BybitOpenOnly | None = None,
        order_filter: BybitOrderFilter | None = None,
        limit: int | None = None,
        cursor: str | None = None,
    ) -> BybitOrderCursorList: ...

class BybitHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        demo: bool = False,
        testnet: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        recv_window_ms: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @property
    def base_url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def cancel_all_requests(self) -> None: ...
    def set_use_spot_position_reports(self, value: bool) -> None: ...
    async def get_account_details(self) -> BybitAccountDetails: ...
    async def set_margin_mode(
        self,
        margin_mode: BybitMarginMode,
    ) -> None: ...
    async def set_leverage(
        self,
        product_type: BybitProductType,
        symbol: str,
        buy_leverage: str,
        sell_leverage: str,
    ) -> None: ...
    async def switch_mode(
        self,
        product_type: BybitProductType,
        mode: BybitPositionMode,
        symbol: str | None = None,
        coin: str | None = None,
    ) -> None: ...
    async def get_spot_borrow_amount(
        self,
        coin: str,
    ) -> Decimal: ...
    async def borrow_spot(
        self,
        coin: str,
        amount: Quantity,
    ) -> Any: ...
    async def repay_spot_borrow(
        self,
        coin: str,
        amount: Quantity | None = None,
    ) -> Any: ...
    async def request_instruments(
        self,
        product_type: BybitProductType,
        symbol: str | None = None,
    ) -> list[Instrument]: ...
    async def request_trades(
        self,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_funding_rates(
        self,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[FundingRateUpdate]: ...
    async def request_orderbook_snapshot(
        self,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        limit: int | None = None,
    ) -> OrderBookDeltas: ...
    async def request_bars(
        self,
        product_type: BybitProductType,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
        timestamp_on_close: bool = True,
    ) -> list[Bar]: ...
    async def request_fee_rates(
        self,
        product_type: BybitProductType,
        symbol: str | None = None,
        base_coin: str | None = None,
    ) -> list[Any]: ...
    async def request_account_state(
        self,
        account_type: BybitAccountType,
        account_id: AccountId,
    ) -> AccountState: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId | None = None,
        open_only: bool = False,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId | None = None,
        start: int | None = None,
        end: int | None = None,
        limit: int | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_reports(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId | None = None,
    ) -> list[PositionStatusReport]: ...
    async def query_order(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> OrderStatusReport | None: ...
    async def submit_order(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
        post_only: bool | None = None,
        reduce_only: bool = False,
        is_quote_quantity: bool = False,
        is_leverage: bool = False,
    ) -> OrderStatusReport: ...
    async def cancel_order(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> OrderStatusReport: ...
    async def cancel_all_orders(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
    ) -> list[OrderStatusReport]: ...
    async def modify_order(
        self,
        account_id: AccountId,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
        quantity: Quantity | None = None,
        price: Price | None = None,
    ) -> OrderStatusReport: ...

class BybitWebSocketClient:
    @staticmethod
    def new_public(
        product_type: BybitProductType,
        environment: BybitEnvironment,
        url: str | None = None,
        heartbeat: int | None = None,
    ) -> BybitWebSocketClient: ...
    @staticmethod
    def new_private(
        environment: BybitEnvironment,
        api_key: str | None = None,
        api_secret: str | None = None,
        url: str | None = None,
        heartbeat: int | None = None,
    ) -> BybitWebSocketClient: ...
    @staticmethod
    def new_trade(
        environment: BybitEnvironment,
        api_key: str | None = None,
        api_secret: str | None = None,
        url: str | None = None,
        heartbeat: int | None = None,
    ) -> BybitWebSocketClient: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def subscription_count(self) -> int: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def set_account_id(self, account_id: AccountId) -> None: ...
    def set_mm_level(self, mm_level: int) -> None: ...
    def set_bars_timestamp_on_close(self, value: bool) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    async def connect(self, callback: Any) -> None: ...
    async def close(self) -> None: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def subscribe(self, topics: list[str]) -> None: ...
    async def unsubscribe(self, topics: list[str]) -> None: ...
    async def subscribe_orderbook(self, instrument_id: InstrumentId, depth: int) -> None: ...
    async def unsubscribe_orderbook(self, instrument_id: InstrumentId, depth: int) -> None: ...
    async def subscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_ticker(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_ticker(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_bars(self, bar_type: BarType) -> None: ...
    async def unsubscribe_bars(self, bar_type: BarType) -> None: ...
    async def subscribe_orders(self) -> None: ...
    async def unsubscribe_orders(self) -> None: ...
    async def subscribe_executions(self) -> None: ...
    async def unsubscribe_executions(self) -> None: ...
    async def subscribe_positions(self) -> None: ...
    async def unsubscribe_positions(self) -> None: ...
    async def subscribe_wallet(self) -> None: ...
    async def unsubscribe_wallet(self) -> None: ...
    async def submit_order(
        self,
        product_type: BybitProductType,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        is_quote_quantity: bool = False,
        time_in_force: TimeInForce | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
        post_only: bool | None = None,
        reduce_only: bool | None = None,
        is_leverage: bool = False,
    ) -> None: ...
    async def modify_order(
        self,
        product_type: BybitProductType,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        venue_order_id: VenueOrderId | None = None,
        quantity: Quantity | None = None,
        price: Price | None = None,
    ) -> None: ...
    async def cancel_order(
        self,
        product_type: BybitProductType,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    async def batch_place_orders(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        orders: list[BybitWsPlaceOrderParams],
    ) -> None: ...
    async def batch_modify_orders(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        orders: list[BybitWsAmendOrderParams],
    ) -> None: ...
    async def batch_cancel_orders(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        orders: list[BybitWsCancelOrderParams],
    ) -> None: ...
    def build_place_order_params(
        self,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        is_quote_quantity: bool = False,
        time_in_force: TimeInForce | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
        post_only: bool | None = None,
        reduce_only: bool | None = None,
        is_leverage: bool = False,
    ) -> BybitWsPlaceOrderParams: ...
    def build_amend_order_params(
        self,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId | None = None,
        client_order_id: ClientOrderId | None = None,
        quantity: Quantity | None = None,
        price: Price | None = None,
    ) -> BybitWsAmendOrderParams: ...
    def build_cancel_order_params(
        self,
        product_type: BybitProductType,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId | None = None,
        client_order_id: ClientOrderId | None = None,
    ) -> BybitWsCancelOrderParams: ...

class BybitWsPlaceOrderParams:
    category: BybitProductType
    symbol: str
    side: str
    order_type: str
    qty: str
    is_leverage: int | None
    market_unit: str | None
    price: str | None
    time_in_force: str | None
    order_link_id: str | None
    reduce_only: bool | None
    close_on_trigger: bool | None
    trigger_price: str | None
    trigger_by: str | None
    trigger_direction: int | None
    tpsl_mode: str | None
    take_profit: str | None
    stop_loss: str | None
    tp_trigger_by: str | None
    sl_trigger_by: str | None
    sl_trigger_price: str | None
    tp_trigger_price: str | None
    sl_order_type: str | None
    tp_order_type: str | None
    sl_limit_price: str | None
    tp_limit_price: str | None

class BybitWsAmendOrderParams:
    category: BybitProductType
    symbol: str
    order_id: str | None
    order_link_id: str | None
    qty: str | None
    price: str | None
    trigger_price: str | None
    take_profit: str | None
    stop_loss: str | None
    tp_trigger_by: str | None
    sl_trigger_by: str | None

class BybitWsCancelOrderParams:
    category: BybitProductType
    symbol: str
    order_id: str | None
    order_link_id: str | None

def get_bybit_http_base_url(environment: BybitEnvironment) -> str: ...
def get_bybit_ws_url_public(
    product_type: BybitProductType, environment: BybitEnvironment
) -> str: ...
def get_bybit_ws_url_private(environment: BybitEnvironment) -> str: ...
def get_bybit_ws_url_trade(environment: BybitEnvironment) -> str: ...
def bybit_extract_raw_symbol(symbol: str) -> str: ...
def bybit_bar_spec_to_interval(aggregation: int, step: int) -> str: ...
def bybit_product_type_from_symbol(symbol: str) -> BybitProductType: ...

# Databento

class DatabentoStatisticType(Enum):
    OPENING_PRICE = "OPENING_PRICE"
    INDICATIVE_OPENING_PRICE = "INDICATIVE_OPENING_PRICE"
    SETTLEMENT_PRICE = "SETTLEMENT_PRICE"
    TRADING_SESSION_LOW_PRICE = "TRADING_SESSION_LOW_PRICE"
    TRADING_SESSION_HIGH_PRICE = "TRADING_SESSION_HIGH_PRICE"
    CLEARED_VOLUME = "CLEARED_VOLUME"
    LOWEST_OFFER = "LOWEST_OFFER"
    HIGHEST_BID = "HIGHEST_BID"
    OPEN_INTEREST = "OPEN_INTEREST"
    FIXING_PRICE = "FIXING_PRICE"
    CLOSE_PRICE = "CLOSE_PRICE"
    NET_CHANGE = "NET_CHANGE"
    VWAP = "VWAP"

class DatabentoStatisticUpdateAction(Enum):
    ADDED = "ADDED"
    DELETED = "DELETED"

class DatabentoPublisher: ...

class DatabentoImbalance:
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def ref_price(self) -> Price: ...
    @property
    def cont_book_clr_price(self) -> Price: ...
    @property
    def auct_interest_clr_price(self) -> Price: ...
    @property
    def paired_qty(self) -> Quantity: ...
    @property
    def total_imbalance_qty(self) -> Quantity: ...
    @property
    def side(self) -> OrderSide: ...
    @property
    def significant_imbalance(self) -> str: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class DatabentoStatistics:
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def stat_type(self) -> DatabentoStatisticType: ...
    @property
    def update_action(self) -> DatabentoStatisticUpdateAction: ...
    @property
    def price(self) -> Price | None: ...
    @property
    def quantity(self) -> Quantity | None: ...
    @property
    def channel_id(self) -> int: ...
    @property
    def stat_flags(self) -> int: ...
    @property
    def sequence(self) -> int: ...
    @property
    def ts_ref(self) -> int: ...
    @property
    def ts_in_delta(self) -> int: ...
    @property
    def ts_event(self) -> int: ...
    @property
    def ts_recv(self) -> int: ...
    @property
    def ts_init(self) -> int: ...

class DatabentoSubscriptionAck:
    @property
    def schema(self) -> str: ...
    @property
    def message(self) -> str: ...
    @property
    def ts_received(self) -> int: ...

class DatabentoDataLoader:
    def __init__(
        self,
        publishers_filepath: PathLike[str] | str | None = None,
    ) -> None: ...
    def load_publishers(self, filepath: PathLike[str] | str) -> None: ...
    def set_dataset_for_venue(self, dataset: str, venue: Venue) -> None: ...
    def get_publishers(self) -> dict[int, DatabentoPublisher]: ...
    def get_dataset_for_venue(self, venue: Venue) -> str: ...
    def schema_for_file(self, filepath: str) -> str: ...
    def load_instruments(self, filepath: str, use_exchange_as_venue: bool) -> list[Instrument]: ...
    def load_order_book_deltas(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[OrderBookDelta]: ...
    def load_order_book_deltas_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
        include_trades: bool | None = None,
    ) -> object: ...
    def load_order_book_depth10(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[OrderBookDepth10]: ...
    def load_order_book_depth10_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> object: ...
    def load_quotes(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[QuoteTick]: ...
    def load_quotes_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
        include_trades: bool | None = None,
    ) -> object: ...
    def load_bbo_quotes(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[QuoteTick]: ...
    def load_bbo_quotes_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> object: ...
    def load_cmbp_quotes(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[QuoteTick]: ...
    def load_cmbp_quotes_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
        include_trades: bool | None = None,
    ) -> object: ...
    def load_cbbo_quotes(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[QuoteTick]: ...
    def load_cbbo_quotes_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> object: ...
    def load_tbbo_trades(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[TradeTick]: ...
    def load_tbbo_trades_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> object: ...
    def load_tcbbo_trades(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[TradeTick]: ...
    def load_tcbbo_trades_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> object: ...
    def load_trades(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[TradeTick]: ...
    def load_trades_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> object: ...
    def load_bars(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
        timestamp_on_close: bool = True,
    ) -> list[Bar]: ...
    def load_bars_as_pycapsule(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
        timestamp_on_close: bool = True,
    ) -> object: ...
    def load_status(
        self, filepath: str, instrument_id: InstrumentId | None = None
    ) -> list[InstrumentStatus]: ...
    def load_imbalance(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[DatabentoImbalance]: ...
    def load_statistics(
        self,
        filepath: str,
        instrument_id: InstrumentId | None = None,
        price_precision: int | None = None,
    ) -> list[DatabentoStatistics]: ...

class DatabentoHistoricalClient:
    def __init__(
        self,
        key: str,
        publishers_filepath: str,
        use_exchange_as_venue: bool,
    ) -> None: ...
    @property
    def key(self) -> str: ...
    async def get_dataset_range(self, dataset: str) -> dict[str, str]: ...
    async def get_range_instruments(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        start: int,
        end: int | None = None,
        limit: int | None = None,
    ) -> list[Instrument]: ...
    async def get_range_quotes(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        start: int,
        end: int | None = None,
        limit: int | None = None,
        price_precision: int | None = None,
        schema: str | None = None,
    ) -> list[QuoteTick]: ...
    async def get_range_trades(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        start: int,
        end: int | None = None,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def get_range_bars(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        aggregation: BarAggregation,
        start: int,
        end: int | None = None,
        limit: int | None = None,
        timestamp_on_close: bool | None = None,
    ) -> list[Bar]: ...
    async def get_order_book_depth10(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        start: int,
        end: int | None = None,
        depth: int | None = None,
    ) -> list[OrderBookDepth10]: ...
    async def get_range_imbalance(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        start: int,
        end: int | None = None,
        limit: int | None = None,
    ) -> list[DatabentoImbalance]: ...
    async def get_range_statistics(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        start: int,
        end: int | None = None,
        limit: int | None = None,
    ) -> list[DatabentoStatistics]: ...
    async def get_range_status(
        self,
        dataset: str,
        instrument_ids: list[InstrumentId],
        start: int,
        end: int | None = None,
        limit: int | None = None,
    ) -> list[InstrumentStatus]: ...

class DatabentoLiveClient:
    def __init__(
        self,
        key: str,
        dataset: str,
        publishers_filepath: str,
        use_exchange_as_venue: bool,
        bars_timestamp_on_close: bool = True,
        reconnect_timeout_mins: int | None = None,
    ) -> None: ...
    @property
    def key(self) -> str: ...
    @property
    def dataset(self) -> str: ...
    def is_running(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def subscribe(
        self,
        schema: str,
        instrument_ids: list[InstrumentId],
        stype_in: str | None = None,
        start: int | None = None,
        snapshot: bool | None = False,
    ) -> dict[str, str]: ...
    def start(
        self,
        callback: Callable,
        callback_pyo3: Callable,
    ) -> Awaitable[None]: ...
    def close(self) -> None: ...

# Deribit

class DeribitHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        is_testnet: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @property
    def is_testnet(self) -> bool: ...
    def is_initialized(self) -> bool: ...
    def cache_instruments(self, instruments: list[Instrument]) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    async def request_instruments(
        self,
        currency: DeribitCurrency,
        product_type: DeribitProductType | None = None,
    ) -> list[Instrument]: ...
    async def request_instrument(self, instrument_id: InstrumentId) -> Instrument: ...
    async def request_trades(
        self,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_bars(
        self,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[Bar]: ...
    async def request_book_snapshot(
        self,
        instrument_id: InstrumentId,
        depth: int | None = None,
    ) -> OrderBook: ...
    async def request_account_state(self, account_id: AccountId) -> AccountState: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
        start: int | None = None,
        end: int | None = None,
        open_only: bool = False,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
        start: int | None = None,
        end: int | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
    ) -> list[PositionStatusReport]: ...

class DeribitWebSocketClient:
    def __init__(
        self,
        url: str | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
        heartbeat_interval: int | None = None,
        is_testnet: bool = False,
    ) -> None: ...
    @staticmethod
    def new_public(is_testnet: bool) -> DeribitWebSocketClient: ...
    @staticmethod
    def with_credentials(
        is_testnet: bool,
        account_id: AccountId | None = None,
    ) -> DeribitWebSocketClient: ...
    @property
    def url(self) -> str: ...
    @property
    def is_testnet(self) -> bool: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def has_credentials(self) -> bool: ...
    def is_authenticated(self) -> bool: ...
    def cancel_all_requests(self) -> None: ...
    def cache_instruments(self, instruments: list[Instrument]) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def set_account_id(self, account_id: AccountId) -> None: ...
    def set_bars_timestamp_on_close(self, value: bool) -> None: ...
    async def connect(
        self,
        instruments: list[Instrument],
        callback: Callable,
    ) -> None: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def close(self) -> None: ...
    async def authenticate(self, session_name: str | None = None) -> None: ...
    async def authenticate_session(self, session_name: str) -> None: ...
    async def subscribe(self, channels: list[str]) -> None: ...
    async def subscribe_book(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
        depth: int | None = None,
    ) -> None: ...
    async def subscribe_book_grouped(
        self,
        instrument_id: InstrumentId,
        group: str,
        depth: int,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def subscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_ticker(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def subscribe_trades(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def subscribe_bars(self, bar_type: BarType) -> None: ...
    async def subscribe_chart(
        self,
        instrument_id: InstrumentId,
        resolution: str,
    ) -> None: ...
    async def subscribe_user_orders(self) -> None: ...
    async def subscribe_user_trades(self) -> None: ...
    async def subscribe_user_portfolio(self) -> None: ...
    async def subscribe_instrument_state(self, kind: str, currency: str) -> None: ...
    async def subscribe_perpetual_interest_rates(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def unsubscribe(self, channels: list[str]) -> None: ...
    async def unsubscribe_book(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
        depth: int | None = None,
    ) -> None: ...
    async def unsubscribe_book_grouped(
        self,
        instrument_id: InstrumentId,
        group: str,
        depth: int,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def unsubscribe_ticker(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def unsubscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_trades(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def unsubscribe_bars(self, bar_type: BarType) -> None: ...
    async def unsubscribe_perpetual_interest_rates(
        self,
        instrument_id: InstrumentId,
        interval: DeribitUpdateInterval | None = None,
    ) -> None: ...
    async def unsubscribe_instrument_state(self, kind: str, currency: str) -> None: ...
    async def unsubscribe_chart(
        self,
        instrument_id: InstrumentId,
        resolution: str,
    ) -> None: ...
    async def unsubscribe_user_orders(self) -> None: ...
    async def unsubscribe_user_trades(self) -> None: ...
    async def unsubscribe_user_portfolio(self) -> None: ...
    async def submit_order(
        self,
        order_side: OrderSide,
        quantity: Quantity,
        order_type: OrderType,
        client_order_id: ClientOrderId,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        price: Price | None = None,
        time_in_force: TimeInForce | None = None,
        post_only: bool = False,
        reduce_only: bool = False,
        trigger_price: Price | None = None,
        trigger: str | None = None,
    ) -> None: ...
    async def modify_order(
        self,
        order_id: str,
        quantity: Quantity,
        price: Price,
        client_order_id: ClientOrderId,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
    ) -> None: ...
    async def cancel_order(
        self,
        order_id: str,
        client_order_id: ClientOrderId,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
    ) -> None: ...
    async def cancel_all_orders(
        self,
        instrument_id: InstrumentId,
        order_type: str | None = None,
    ) -> None: ...
    async def query_order(
        self,
        order_id: str,
        client_order_id: ClientOrderId,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
    ) -> None: ...

def get_deribit_http_base_url(is_testnet: bool) -> str: ...
def get_deribit_ws_url(is_testnet: bool) -> str: ...

class DeribitCurrency(Enum):
    BTC = "BTC"
    ETH = "ETH"
    USDC = "USDC"
    USDT = "USDT"
    EURR = "EURR"
    ANY = "ANY"

class DeribitProductType(Enum):
    FUTURE = "FUTURE"
    OPTION = "OPTION"
    SPOT = "SPOT"
    FUTURE_COMBO = "FUTURE_COMBO"
    OPTION_COMBO = "OPTION_COMBO"

class DeribitUpdateInterval(Enum):
    RAW = "RAW"
    MS100 = "MS100"
    AGG2 = "AGG2"

    @classmethod
    def from_str(cls, value: str) -> DeribitUpdateInterval: ...

# Tardis

def tardis_normalize_symbol_str(
    symbol: str, exchange: str, instrument_type: str, is_inverse: bool | None = None
) -> str: ...
def tardis_exchanges() -> list[str]: ...
def tardis_exchange_from_venue_str(value: str) -> list[str]: ...
def tardis_exchange_to_venue_str(value: str) -> str: ...
def tardis_exchange_is_option_exchange(value: str) -> bool: ...
def bar_spec_to_tardis_trade_bar_string(bar_spec: BarSpecification) -> str: ...
def load_tardis_deltas(
    filepath: str,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> list[OrderBookDelta]: ...
def load_tardis_quotes(
    filepath: str,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> list[QuoteTick]: ...
def load_tardis_trades(
    filepath: str,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> list[TradeTick]: ...
def load_tardis_depth10_from_snapshot5(
    filepath: str,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> list[OrderBookDepth10]: ...
def load_tardis_depth10_from_snapshot25(
    filepath: str,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> list[OrderBookDepth10]: ...
def stream_tardis_deltas(
    filepath: str,
    chunk_size: int = 100_000,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> Iterator[list[OrderBookDelta]]: ...
def stream_tardis_batched_deltas(
    filepath: str,
    chunk_size: int = 100_000,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> Iterator[list[object]]: ...
def stream_tardis_quotes(
    filepath: str,
    chunk_size: int = 100_000,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> Iterator[list[QuoteTick]]: ...
def stream_tardis_trades(
    filepath: str,
    chunk_size: int = 100_000,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> Iterator[list[TradeTick]]: ...
def stream_tardis_depth10_from_snapshot5(
    filepath: str,
    chunk_size: int = 100_000,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> Iterator[list[OrderBookDepth10]]: ...
def stream_tardis_depth10_from_snapshot25(
    filepath: str,
    chunk_size: int = 100_000,
    price_precision: int | None = None,
    size_precision: int | None = None,
    instrument_id: InstrumentId | None = None,
    limit: int | None = None,
) -> Iterator[list[OrderBookDepth10]]: ...

class TardisInstrumentMiniInfo:
    def __init__(
        self,
        instrument_id: InstrumentId,
        raw_symbol: str,
        exchange: str,
        price_precision: int,
        size_precision: int,
    ) -> None: ...
    @property
    def instrument_id(self) -> InstrumentId: ...
    @property
    def raw_symbol(self) -> str: ...
    @property
    def exchange(self) -> str: ...
    @property
    def price_precision(self) -> int: ...
    @property
    def size_precision(self) -> int: ...

class TardisHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        base_url: str | None = None,
        timeout_secs: int = 60,
        normalize_symbols: bool = True,
    ) -> None: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    async def instruments(
        self,
        exchange: str,
        symbol: str | None = None,
        base_currency: list[str] | None = None,
        quote_currency: list[str] | None = None,
        instrument_type: list[str] | None = None,
        contract_type: list[str] | None = None,
        active: bool | None = None,
        start: int | None = None,
        end: int | None = None,
        available_offset: int | None = None,
        effective: int | None = None,
        ts_init: int | None = None,
    ) -> list[Instrument]: ...

class ReplayNormalizedRequestOptions:
    @classmethod
    def from_json(cls, data: bytes) -> ReplayNormalizedRequestOptions: ...
    @classmethod
    def from_json_array(cls, data: bytes) -> list[ReplayNormalizedRequestOptions]: ...

class StreamNormalizedRequestOptions:
    @classmethod
    def from_json(cls, data: bytes) -> StreamNormalizedRequestOptions: ...
    @classmethod
    def from_json_array(cls, data: bytes) -> list[StreamNormalizedRequestOptions]: ...

class TardisMachineClient:
    def __init__(self, base_url: str | None = None, normalize_symbols: bool = True) -> None: ...
    def is_closed(self) -> bool: ...
    def close(self) -> None: ...
    def stream(
        self,
        instruments: list[TardisInstrumentMiniInfo],
        options: list[StreamNormalizedRequestOptions],
        callback: Callable,
    ) -> Awaitable[None]: ...
    def replay(
        self,
        instruments: list[TardisInstrumentMiniInfo],
        options: list[ReplayNormalizedRequestOptions],
        callback: Callable,
    ) -> Awaitable[None]: ...
    def replay_bars(
        self,
        instruments: list[TardisInstrumentMiniInfo],
        options: list[ReplayNormalizedRequestOptions],
    ) -> Awaitable[list[Bar]]: ...

async def run_tardis_machine_replay(
    config_filepath: str, output_path: str | None = None
) -> None: ...

# Coinbase International

class CoinbaseIntxHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        api_passphrase: str | None = None,
        base_url: str | None = None,
        timeout_secs: int | None = None,
    ) -> None: ...
    @property
    def base_url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def is_initialized(self) -> bool: ...
    def get_cached_symbols(self) -> list[str]: ...
    def add_instrument(self, instrument: Instrument) -> None: ...
    async def list_portfolios(self) -> list[dict[str, Any]]: ...
    async def request_account_state(self, account_id: AccountId) -> AccountState: ...
    async def request_instruments(self) -> list[Instrument]: ...
    async def request_instrument(self, symbol: Symbol) -> Instrument: ...
    async def request_order_status_report(
        self,
        account_id: AccountId,
        venue_order_id: VenueOrderId,
    ) -> OrderStatusReport: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
        symbol: Symbol,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
        client_order_id: ClientOrderId | None = None,
        start: dt.datetime | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_report(
        self,
        account_id: AccountId,
        symbol: Symbol,
    ) -> PositionStatusReport: ...
    async def request_position_status_reports(
        self,
        account_id: AccountId,
    ) -> list[PositionStatusReport]: ...
    async def submit_order(
        self,
        account_id: AccountId,
        symbol: Symbol,
        client_order_id: ClientOrderId,
        order_type: OrderType,
        order_side: OrderSide,
        quantity: Quantity,
        time_in_force: TimeInForce,
        expire_time: dt.datetime | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
        trigger_type: TriggerType | None = None,
        post_only: bool | None = None,
        reduce_only: bool | None = None,
    ) -> OrderStatusReport: ...
    async def cancel_order(
        self,
        account_id: AccountId,
        client_order_id: ClientOrderId,
    ) -> OrderStatusReport: ...
    async def cancel_orders(
        self,
        account_id: AccountId,
        symbol: Symbol,
        order_side: OrderSide | None = None,
    ) -> list[OrderStatusReport]: ...
    async def modify_order(
        self,
        account_id: AccountId,
        client_order_id: ClientOrderId,
        new_client_order_id: ClientOrderId,
        price: Price | None = None,
        trigger_price: Price | None = None,
        quantity: Quantity | None = None,
    ) -> OrderStatusReport: ...

class CoinbaseIntxWebSocketClient:
    def __init__(
        self,
        url: str | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
        api_passphrase: str | None = None,
        heartbeat: int | None = None,
    ) -> None: ...
    @property
    def url(self) -> str: ...
    @property
    def api_key(self) -> str: ...
    @property
    def api_key_masked(self) -> str: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def get_subscriptions(self, instrument_id: InstrumentId) -> list[str]: ...
    async def connect(
        self,
        instruments: list[Instrument],
        callback: Callable,
    ) -> Awaitable: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def close(self) -> None: ...
    async def subscribe_instruments(
        self, instrument_ids: list[InstrumentId] | None = None
    ) -> None: ...
    async def subscribe_book(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def subscribe_quotes(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def subscribe_trades(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def subscribe_mark_prices(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def subscribe_index_prices(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def subscribe_bars(self, bar_type: BarType) -> None: ...
    async def unsubscribe_instruments(self) -> None: ...
    async def unsubscribe_book(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def unsubscribe_quotes(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def unsubscribe_trades(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def unsubscribe_mark_prices(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def unsubscribe_index_prices(self, instrument_ids: list[InstrumentId]) -> None: ...
    async def unsubscribe_bars(self, bar_type: BarType) -> None: ...

class CoinbaseIntxFixClient:
    def __init__(
        self,
        endpoint: str | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
        api_passphrase: str | None = None,
        portfolio_id: str | None = None,
    ) -> None: ...
    @property
    def endpoint(self) -> str: ...
    @property
    def api_key(self) -> str: ...
    @property
    def api_key_masked(self) -> str: ...
    @property
    def portfolio_id(self) -> str: ...
    @property
    def sender_comp_id(self) -> str: ...
    @property
    def target_comp_id(self) -> str: ...
    def is_connected(self) -> bool: ...
    def is_logged_on(self) -> bool: ...
    async def connect(self, handler: object) -> Awaitable: ...
    async def close(self) -> None: ...

# OKX

class OKXHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        api_passphrase: str | None = None,
        base_url: str | None = None,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        is_demo: bool = False,
        proxy_url: str | None = None,
    ) -> None: ...
    @staticmethod
    def from_env() -> OKXHttpClient: ...
    @property
    def base_url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def is_initialized(self) -> bool: ...
    def get_cached_symbols(self) -> list[str]: ...
    def cache_instruments(self, instruments: list[Instrument]) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def cancel_all_requests(self) -> None: ...
    async def set_position_mode(self, mode: OKXPositionMode) -> None: ...
    async def request_instrument(self, instrument_id: InstrumentId) -> Instrument: ...
    async def request_instruments(
        self, instrument_type: OKXInstrumentType, instrument_family: str | None = None
    ) -> tuple[list[Instrument], list[tuple[str, int]]]: ...
    async def request_account_state(self, account_id: AccountId) -> AccountState: ...
    async def request_trades(
        self,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_bars(
        self,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[Bar]: ...
    async def request_mark_price(
        self,
        instrument_id: InstrumentId,
    ) -> MarkPriceUpdate: ...
    async def request_index_price(
        self,
        instrument_id: InstrumentId,
    ) -> IndexPriceUpdate: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
        instrument_type: OKXInstrumentType | None = None,
        instrument_id: InstrumentId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        open_only: bool = False,
        limit: int | None = None,
    ) -> list[OrderStatusReport]: ...
    async def request_algo_order_status_reports(
        self,
        account_id: AccountId,
        instrument_type: OKXInstrumentType | None = None,
        instrument_id: InstrumentId | None = None,
        algo_id: str | None = None,
        algo_client_order_id: ClientOrderId | None = None,
        state: OKXOrderStatus | None = None,
        limit: int | None = None,
    ) -> list[OrderStatusReport]: ...
    async def request_algo_order_status_report(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
    ) -> OrderStatusReport | None: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
        instrument_type: OKXInstrumentType | None = None,
        instrument_id: InstrumentId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_reports(
        self,
        account_id: AccountId,
        instrument_type: OKXInstrumentType | None = None,
        instrument_id: InstrumentId | None = None,
    ) -> list[PositionStatusReport]: ...
    async def place_algo_order(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        td_mode: OKXTradeMode,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        trigger_price: Price,
        trigger_type: TriggerType | None = None,
        limit_price: Price | None = None,
        reduce_only: bool | None = None,
    ) -> Any: ...
    async def cancel_algo_order(
        self,
        instrument_id: InstrumentId,
        algo_id: str,
    ) -> Any: ...
    async def cancel_algo_orders(
        self,
        orders: list[tuple[InstrumentId, str]],
    ) -> list[Any]: ...
    async def get_server_time(self) -> int: ...
    async def get_balance(self) -> list[OKXBalanceDetail]: ...

class OKXBalanceDetail:
    @property
    def ccy(self) -> str: ...
    @property
    def cash_bal(self) -> str: ...
    @property
    def liab(self) -> str: ...

class OKXWebSocketClient:
    def __init__(
        self,
        url: str | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
        api_passphrase: str | None = None,
        account_id: AccountId | None = None,
        heartbeat: int | None = None,
    ) -> None: ...
    @staticmethod
    def with_credentials(
        url: str | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
        api_passphrase: str | None = None,
        account_id: AccountId | None = None,
        heartbeat: int | None = None,
    ) -> OKXWebSocketClient: ...
    @staticmethod
    def from_env() -> OKXWebSocketClient: ...
    @property
    def url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    @property
    def vip_level(self) -> OKXVipLevel: ...
    def set_vip_level(self, vip_level: OKXVipLevel) -> None: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def cache_instruments(self, instruments: list[Instrument]) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def cache_inst_id_codes(self, mappings: list[tuple[str, int]]) -> None: ...
    def cancel_all_requests(self) -> None: ...
    def get_subscriptions(self, instrument_id: InstrumentId) -> list[str]: ...
    async def connect(
        self,
        instruments: list[Instrument],
        callback: Callable,
    ) -> Awaitable: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def close(self) -> None: ...
    async def subscribe_instruments(self, instrument_type: OKXInstrumentType) -> None: ...
    async def subscribe_instrument(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book_depth5(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book50_l2_tbt(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book_l2_tbt(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book_with_depth(self, instrument_id: InstrumentId, depth: int) -> None: ...
    async def subscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_trades(self, instrument_id: InstrumentId, aggregated: bool) -> None: ...
    async def subscribe_bars(self, bar_type: BarType) -> None: ...
    async def unsubscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book_depth5(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book50_l2_tbt(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book_l2_tbt(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_trades(self, instrument_id: InstrumentId, aggregated: bool) -> None: ...
    async def unsubscribe_bars(self, bar_type: BarType) -> None: ...
    async def subscribe_ticker(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_ticker(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_mark_prices(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_mark_prices(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_index_prices(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_index_prices(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_funding_rates(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_funding_rates(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_orders(self, instrument_type: OKXInstrumentType) -> None: ...
    async def unsubscribe_orders(self, instrument_type: OKXInstrumentType) -> None: ...
    async def subscribe_orders_algo(self, instrument_type: OKXInstrumentType) -> None: ...
    async def unsubscribe_orders_algo(self, instrument_type: OKXInstrumentType) -> None: ...
    async def subscribe_fills(self, instrument_type: OKXInstrumentType) -> None: ...
    async def unsubscribe_fills(self, instrument_type: OKXInstrumentType) -> None: ...
    async def subscribe_account(self) -> None: ...
    async def unsubscribe_account(self) -> None: ...
    async def submit_order(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        td_mode: OKXTradeMode,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
        post_only: bool | None = None,
        reduce_only: bool | None = None,
        quote_quantity: bool | None = None,
        position_side: PositionSide | None = None,
    ) -> None: ...
    async def cancel_order(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    async def modify_order(
        self,
        trader_id: TraderId,
        strategy_id: StrategyId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
        price: Price | None = None,
        quantity: Quantity | None = None,
    ) -> None: ...
    async def batch_submit_orders(
        self,
        orders: list[Any],
    ) -> None: ...
    async def batch_cancel_orders(
        self,
        orders: list[Any],
    ) -> None: ...
    async def mass_cancel_orders(
        self,
        instrument_id: InstrumentId,
    ) -> None: ...
    async def batch_modify_orders(
        self,
        orders: list[Any],
    ) -> None: ...

class OKXWebSocketError:
    @property
    def code(self) -> str: ...
    @property
    def message(self) -> str: ...
    @property
    def conn_id(self) -> str | None: ...
    @property
    def ts_event(self) -> int: ...

class OKXEndpointType(Enum):
    Public = "Public"
    Private = "Private"
    Business = "Business"

def get_okx_http_base_url() -> str: ...
def get_okx_ws_url_public(is_demo: bool) -> str: ...
def get_okx_ws_url_private(is_demo: bool) -> str: ...
def get_okx_ws_url_business(is_demo: bool) -> str: ...
def okx_requires_authentication(endpoint_type: OKXEndpointType) -> bool: ...

class OKXInstrumentType(Enum):
    ANY = "ANY"
    SPOT = "SPOT"
    MARGIN = "MARGIN"
    SWAP = "SWAP"
    FUTURES = "FUTURES"
    OPTION = "OPTION"

class OKXContractType(Enum):
    NONE = "NONE"
    LINEAR = "LINEAR"
    INVERSE = "INVERSE"

class OKXMarginMode(Enum):
    NONE = "NONE"
    ISOLATED = "ISOLATED"
    CROSS = "CROSS"

class OKXTradeMode(Enum):
    CASH = "CASH"
    ISOLATED = "ISOLATED"
    CROSS = "CROSS"
    SPOT_ISOLATED = "SPOT_ISOLATED"

class OKXPositionMode(Enum):
    NET_MODE = "NET_MODE"
    LONG_SHORT_MODE = "LONG_SHORT_MODE"

class OKXOrderStatus(Enum):
    CANCELED = ...
    LIVE = ...
    EFFECTIVE = ...
    PARTIALLY_FILLED = ...
    FILLED = ...
    MMP_CANCELED = ...
    ORDER_PLACED = ...

class OKXVipLevel(Enum):
    VIP0 = "VIP0"
    VIP1 = "VIP1"
    VIP2 = "VIP2"
    VIP3 = "VIP3"
    VIP4 = "VIP4"
    VIP5 = "VIP5"
    VIP6 = "VIP6"
    VIP7 = "VIP7"
    VIP8 = "VIP8"
    VIP9 = "VIP9"

    @property
    def value(self) -> int: ...  # type: ignore[override]

# BitMEX

def get_bitmex_http_base_url(testnet: bool) -> str: ...
def get_bitmex_ws_url(testnet: bool) -> str: ...

class BitmexHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        testnet: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        recv_window_ms: int | None = None,
        max_requests_per_second: int | None = None,
        max_requests_per_minute: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @staticmethod
    def from_env() -> BitmexHttpClient: ...
    @property
    def base_url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def cancel_all_requests(self) -> None: ...
    async def get_server_time(self) -> int: ...
    async def get_margin(self, currency: str) -> int: ...
    async def get_account_number(self) -> int | None: ...
    async def update_position_leverage(
        self,
        symbol: str,
        leverage: float,
    ) -> PositionStatusReport: ...
    async def request_instrument(
        self,
        instrument_id: InstrumentId,
    ) -> Instrument | None: ...
    async def request_instruments(
        self,
        active_only: bool,
    ) -> list[Instrument]: ...
    async def request_trades(
        self,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_bars(
        self,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
        partial: bool = False,
    ) -> list[Bar]: ...
    async def request_account_state(
        self,
        account_id: AccountId,
    ) -> AccountState: ...
    async def query_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> OrderStatusReport | None: ...
    async def request_order_status_reports(
        self,
        instrument_id: InstrumentId | None = None,
        open_only: bool = False,
        limit: int | None = None,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        instrument_id: InstrumentId | None = None,
        limit: int | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_reports(self) -> list[PositionStatusReport]: ...
    async def submit_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        price: Price | None = None,
        trigger_price: Price | None = None,
        trigger_type: TriggerType | None = None,
        trailing_offset: float | None = None,
        trailing_offset_type: TrailingOffsetType | None = None,
        display_qty: Quantity | None = None,
        post_only: bool = False,
        reduce_only: bool = False,
        order_list_id: OrderListId | None = None,
        contingency_type: ContingencyType | None = None,
    ) -> OrderStatusReport: ...
    async def cancel_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> OrderStatusReport: ...
    async def cancel_orders(
        self,
        instrument_id: InstrumentId,
        client_order_ids: list[ClientOrderId] | None = None,
        venue_order_ids: list[VenueOrderId] | None = None,
    ) -> list[OrderStatusReport]: ...
    async def cancel_all_orders(
        self,
        instrument_id: InstrumentId,
        order_side: OrderSide | None,
    ) -> list[OrderStatusReport]: ...
    async def modify_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
        quantity: Quantity | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
    ) -> OrderStatusReport: ...

class BitmexWebSocketClient:
    def __init__(
        self,
        url: str | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
        account_id: AccountId | None = None,
        heartbeat: int | None = None,
        testnet: bool = False,
    ) -> None: ...
    @property
    def url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def set_account_id(self, account_id: AccountId) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def get_subscriptions(self, instrument_id: InstrumentId) -> list[str]: ...
    async def connect(self, instruments: list[Any], callback: Callable) -> None: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def close(self) -> None: ...
    async def subscribe_instruments(self) -> None: ...
    async def subscribe_instrument(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book_25(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book_depth10(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_mark_prices(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_index_prices(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_funding_rates(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_bars(self, bar_type: BarType) -> None: ...
    async def subscribe_orders(self) -> None: ...
    async def subscribe_executions(self) -> None: ...
    async def subscribe_positions(self) -> None: ...
    async def subscribe_margin(self) -> None: ...
    async def subscribe_wallet(self) -> None: ...
    async def unsubscribe_instruments(self) -> None: ...
    async def unsubscribe_instrument(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book_25(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book_depth10(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_mark_prices(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_index_prices(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_funding_rates(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_bars(self, bar_type: BarType) -> None: ...
    async def unsubscribe_orders(self) -> None: ...
    async def unsubscribe_executions(self) -> None: ...
    async def unsubscribe_positions(self) -> None: ...
    async def unsubscribe_margin(self) -> None: ...
    async def unsubscribe_wallet(self) -> None: ...

class SubmitBroadcaster:
    def __init__(
        self,
        pool_size: int,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        testnet: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        recv_window_ms: int | None = None,
        max_requests_per_second: int | None = None,
        max_requests_per_minute: int | None = None,
        health_check_interval_secs: int = 30,
        health_check_timeout_secs: int = 5,
        expected_reject_patterns: list[str] | None = None,
    ) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def get_metrics(self) -> dict[str, int]: ...
    def get_client_stats(self) -> list[dict[str, Any]]: ...
    async def start(self) -> None: ...
    async def stop(self) -> None: ...
    async def broadcast_submit(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        price: Price | None = None,
        trigger_price: Price | None = None,
        trigger_type: TriggerType | None = None,
        trailing_offset: float | None = None,
        trailing_offset_type: TrailingOffsetType | None = None,
        display_qty: Quantity | None = None,
        post_only: bool = False,
        reduce_only: bool = False,
        order_list_id: OrderListId | None = None,
        contingency_type: ContingencyType | None = None,
        submit_tries: int | None = None,
    ) -> OrderStatusReport: ...

class CancelBroadcaster:
    def __init__(
        self,
        pool_size: int,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        testnet: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        recv_window_ms: int | None = None,
        max_requests_per_second: int | None = None,
        max_requests_per_minute: int | None = None,
        health_check_interval_secs: int = 30,
        health_check_timeout_secs: int = 5,
        expected_reject_patterns: list[str] | None = None,
        idempotent_success_patterns: list[str] | None = None,
    ) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def get_metrics(self) -> dict[str, int]: ...
    def get_client_stats(self) -> list[dict[str, Any]]: ...
    async def start(self) -> None: ...
    async def stop(self) -> None: ...
    async def broadcast_cancel(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> OrderStatusReport | None: ...
    async def broadcast_batch_cancel(
        self,
        instrument_id: InstrumentId,
        client_order_ids: list[ClientOrderId] | None = None,
        venue_order_ids: list[VenueOrderId] | None = None,
    ) -> list[OrderStatusReport]: ...
    async def broadcast_cancel_all(
        self,
        instrument_id: InstrumentId,
        order_side: OrderSide | None = None,
    ) -> list[OrderStatusReport]: ...

# Hyperliquid

HYPERLIQUID_POST_ONLY_WOULD_MATCH: Final[str]
HYPERLIQUID_BUILDER_FEE_NOT_APPROVED: Final[str]

def hyperliquid_product_type_from_symbol(symbol: str) -> HyperliquidProductType: ...
def hyperliquid_cloid_from_client_order_id(client_order_id: ClientOrderId) -> str: ...
def get_hyperliquid_builder_fee_info() -> dict: ...
def print_hyperliquid_builder_fee_info() -> None: ...
def approve_hyperliquid_builder_fee(non_interactive: bool = False) -> bool: ...
def revoke_hyperliquid_builder_fee(non_interactive: bool = False) -> bool: ...
def verify_hyperliquid_builder_fee(wallet_address: str | None = None) -> bool: ...
def get_hyperliquid_http_base_url(is_testnet: bool = False) -> str: ...
def get_hyperliquid_ws_url(is_testnet: bool = False) -> str: ...

class HyperliquidProductType(Enum):
    PERP = "PERP"
    SPOT = "SPOT"

    @classmethod
    def from_str(cls, value: str) -> HyperliquidProductType: ...

class HyperliquidTriggerPriceType(Enum):
    LAST = "LAST"
    MARK = "MARK"
    ORACLE = "ORACLE"

    @classmethod
    def from_str(cls, value: str) -> HyperliquidTriggerPriceType: ...

class HyperliquidTpSl(Enum):
    TP = "TP"
    SL = "SL"

    @classmethod
    def from_str(cls, value: str) -> HyperliquidTpSl: ...

class HyperliquidConditionalOrderType(Enum):
    STOP_MARKET = "STOP_MARKET"
    STOP_LIMIT = "STOP_LIMIT"
    TAKE_PROFIT_MARKET = "TAKE_PROFIT_MARKET"
    TAKE_PROFIT_LIMIT = "TAKE_PROFIT_LIMIT"
    TRAILING_STOP_MARKET = "TRAILING_STOP_MARKET"
    TRAILING_STOP_LIMIT = "TRAILING_STOP_LIMIT"

    @classmethod
    def from_str(cls, value: str) -> HyperliquidConditionalOrderType: ...

class HyperliquidTrailingOffsetType(Enum):
    PRICE = "PRICE"
    PERCENTAGE = "PERCENTAGE"
    BASIS_POINTS = "BASIS_POINTS"

    @classmethod
    def from_str(cls, value: str) -> HyperliquidTrailingOffsetType: ...

class HyperliquidHttpClient:
    def __init__(
        self,
        private_key: str | None = None,
        vault_address: str | None = None,
        is_testnet: bool = False,
        timeout_secs: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @staticmethod
    def from_env() -> HyperliquidHttpClient: ...
    @staticmethod
    def from_credentials(
        private_key: str,
        vault_address: str | None = None,
        is_testnet: bool = False,
        timeout_secs: int | None = None,
        proxy_url: str | None = None,
    ) -> HyperliquidHttpClient: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def set_account_id(self, account_id: str) -> None: ...
    def get_user_address(self) -> str: ...
    def get_spot_fill_coin_mapping(self) -> dict[str, str]: ...
    async def get_perp_meta(self) -> str: ...
    async def get_spot_meta(self) -> str: ...
    async def load_instrument_definitions(
        self,
        include_perp: bool = True,
        include_spot: bool = True,
    ) -> list[Instrument]: ...
    async def request_order_status_reports(
        self,
        instrument_id: str | None = None,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        instrument_id: str | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_reports(
        self,
        instrument_id: str | None = None,
    ) -> list[PositionStatusReport]: ...
    async def request_quote_ticks(
        self,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[QuoteTick]: ...
    async def request_trade_ticks(
        self,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_bars(
        self,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[Bar]: ...
    async def submit_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        price: Price | None = None,
        trigger_price: Price | None = None,
        post_only: bool = False,
        reduce_only: bool = False,
    ) -> OrderStatusReport: ...
    async def submit_orders(
        self,
        orders: list[Order],
    ) -> list[OrderStatusReport]: ...
    async def cancel_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    async def request_account_state(self) -> AccountState: ...

class HyperliquidWebSocketClient:
    def __init__(
        self,
        url: str | None = None,
        testnet: bool = False,
        product_type: HyperliquidProductType = ...,
        account_id: str | None = None,
    ) -> None: ...
    @property
    def url(self) -> str: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def cache_spot_fill_coins(self, mapping: dict[str, str]) -> None: ...
    def cache_cloid_mapping(self, cloid: str, client_order_id: ClientOrderId) -> None: ...
    def remove_cloid_mapping(self, cloid: str) -> None: ...
    def clear_cloid_cache(self) -> None: ...
    def cloid_cache_len(self) -> int: ...
    def get_cloid_mapping(self, cloid: str) -> ClientOrderId | None: ...
    async def connect(self, instruments: list[Any], callback: Callable) -> None: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def close(self) -> None: ...
    async def subscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_book_deltas(
        self,
        instrument_id: InstrumentId,
        book_type: int,
        depth: int,
    ) -> None: ...
    async def subscribe_book_snapshots(
        self,
        instrument_id: InstrumentId,
        book_type: int,
        depth: int,
    ) -> None: ...
    async def subscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_bars(self, bar_type: BarType) -> None: ...
    async def subscribe_mark_prices(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_index_prices(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_funding_rates(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_order_updates(self, user: str) -> None: ...
    async def subscribe_user_events(self, user: str) -> None: ...
    async def subscribe_user_fills(self, user: str) -> None: ...
    async def unsubscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book_deltas(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_bars(self, bar_type: BarType) -> None: ...
    async def unsubscribe_mark_prices(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_index_prices(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_funding_rates(self, instrument_id: InstrumentId) -> None: ...

# Kraken

class KrakenEnvironment(Enum):
    MAINNET = "mainnet"
    DEMO = "demo"

class KrakenProductType(Enum):
    SPOT = "spot"
    FUTURES = "futures"

class KrakenSpotHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        demo: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        proxy_url: str | None = None,
        max_requests_per_second: int | None = None,
    ) -> None: ...
    @property
    def base_url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def cancel_all_requests(self) -> None: ...
    def set_use_spot_position_reports(self, value: bool) -> None: ...
    def set_spot_positions_quote_currency(self, currency: str) -> None: ...
    async def get_server_time(self) -> str: ...
    async def request_instruments(
        self,
        pairs: list[str] | None = None,
    ) -> list[Instrument]: ...
    async def request_trades(
        self,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_bars(
        self,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[Bar]: ...
    async def request_account_state(self, account_id: AccountId) -> AccountState: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        open_only: bool = False,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
    ) -> list[PositionStatusReport]: ...
    async def submit_order(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        expire_time: int | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
        reduce_only: bool = False,
        post_only: bool = False,
    ) -> VenueOrderId: ...
    async def modify_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
        quantity: Quantity | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
    ) -> VenueOrderId: ...
    async def cancel_order(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    async def cancel_all_orders(self) -> int: ...
    async def cancel_orders_batch(
        self,
        venue_order_ids: list[VenueOrderId],
    ) -> int: ...

class KrakenFuturesHttpClient:
    def __init__(
        self,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        demo: bool = False,
        timeout_secs: int | None = None,
        max_retries: int | None = None,
        retry_delay_ms: int | None = None,
        retry_delay_max_ms: int | None = None,
        proxy_url: str | None = None,
        max_requests_per_second: int | None = None,
    ) -> None: ...
    @property
    def base_url(self) -> str: ...
    @property
    def api_key(self) -> str | None: ...
    @property
    def api_key_masked(self) -> str | None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def cancel_all_requests(self) -> None: ...
    async def request_mark_price(self, instrument_id: InstrumentId) -> float: ...
    async def request_index_price(self, instrument_id: InstrumentId) -> float: ...
    async def request_instruments(self) -> list[Instrument]: ...
    async def request_trades(
        self,
        instrument_id: InstrumentId,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_bars(
        self,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[Bar]: ...
    async def request_account_state(
        self,
        account_id: AccountId,
    ) -> AccountState: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        open_only: bool = False,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
    ) -> list[FillReport]: ...
    async def request_position_status_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
    ) -> list[PositionStatusReport]: ...
    async def submit_order(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        price: Price | None = None,
        trigger_price: Price | None = None,
        reduce_only: bool = False,
        post_only: bool = False,
    ) -> OrderStatusReport: ...
    async def modify_order(
        self,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
        quantity: Quantity | None = None,
        price: Price | None = None,
        trigger_price: Price | None = None,
    ) -> VenueOrderId: ...
    async def cancel_order(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId | None = None,
        venue_order_id: VenueOrderId | None = None,
    ) -> None: ...
    async def cancel_all_orders(
        self,
        instrument_id: InstrumentId | None = None,
    ) -> int: ...
    async def cancel_orders_batch(
        self,
        venue_order_ids: list[VenueOrderId],
    ) -> int: ...

class KrakenSpotWebSocketClient:
    def __init__(
        self,
        environment: KrakenEnvironment | None = None,
        private: bool = False,
        base_url: str | None = None,
        heartbeat_secs: int | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
    ) -> None: ...
    @property
    def url(self) -> str: ...
    def is_connected(self) -> bool: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def get_subscriptions(self) -> list[str]: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def set_account_id(self, account_id: AccountId) -> None: ...
    def cache_client_order(
        self,
        client_order_id: ClientOrderId,
        venue_order_id: VenueOrderId | None,
        instrument_id: InstrumentId,
        trader_id: TraderId,
        strategy_id: StrategyId,
    ) -> None: ...
    def cancel_all_requests(self) -> None: ...
    async def connect(self, instruments: list[Instrument], callback: Callable) -> None: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def authenticate(self) -> None: ...
    async def send_ping(self) -> None: ...
    async def disconnect(self) -> None: ...
    async def close(self) -> None: ...
    async def subscribe_book(
        self,
        instrument_id: InstrumentId,
        depth: int | None = None,
    ) -> None: ...
    async def subscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_bars(self, bar_type: BarType) -> None: ...
    async def subscribe_executions(
        self,
        snap_orders: bool = True,
        snap_trades: bool = True,
    ) -> None: ...
    async def unsubscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_bars(self, bar_type: BarType) -> None: ...

class KrakenFuturesWebSocketClient:
    def __init__(
        self,
        environment: KrakenEnvironment | None = None,
        base_url: str | None = None,
        heartbeat_secs: int | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
    ) -> None: ...
    @property
    def has_credentials(self) -> bool: ...
    @property
    def url(self) -> str: ...
    def is_closed(self) -> bool: ...
    def is_active(self) -> bool: ...
    async def wait_until_active(self, timeout_secs: float) -> None: ...
    async def authenticate(self) -> None: ...
    def cache_instruments(self, instruments: list[Instrument]) -> None: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    def set_account_id(self, account_id: AccountId) -> None: ...
    def cache_client_order(
        self,
        client_order_id: ClientOrderId,
        venue_order_id: VenueOrderId | None,
        instrument_id: InstrumentId,
        trader_id: TraderId,
        strategy_id: StrategyId,
    ) -> None: ...
    def sign_challenge(self, challenge: str) -> str: ...
    async def connect(
        self, instruments: list[Instrument], callback: Callable[..., Any]
    ) -> None: ...
    async def disconnect(self) -> None: ...
    async def close(self) -> None: ...
    async def authenticate_with_challenge(self, challenge: str) -> None: ...
    async def set_auth_credentials(
        self, original_challenge: str, signed_challenge: str
    ) -> None: ...
    async def subscribe_book(
        self, instrument_id: InstrumentId, depth: int | None = None
    ) -> None: ...
    async def subscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_mark_price(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_index_price(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_funding_rate(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_book(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_quotes(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_trades(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_mark_price(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_index_price(self, instrument_id: InstrumentId) -> None: ...
    async def unsubscribe_funding_rate(self, instrument_id: InstrumentId) -> None: ...
    async def subscribe_open_orders(self) -> None: ...
    async def subscribe_fills(self) -> None: ...
    async def subscribe_executions(self) -> None: ...

def kraken_product_type_from_symbol(symbol: str) -> KrakenProductType: ...

# Greeks

class BlackScholesGreeksResult:
    price: float
    delta: float
    gamma: float
    vega: float
    theta: float

def black_scholes_greeks(
    s: float,
    r: float,
    b: float,
    vol: float,
    is_call: bool,
    k: float,
    t: float,
    multiplier: float,
) -> BlackScholesGreeksResult:
    """
    Calculate the Black-Scholes Greeks for a given option contract.

    Parameters
    ----------
    s : float
        The current price of the underlying asset.
    r : float
        The risk-free interest rate.
    b : float
        The cost of carry of the underlying asset.
    vol : float
        The volatility of the underlying asset.
    is_call : bool
        Whether the option is a call (True) or a put (False).
    k : float
        The strike price of the option.
    t : float
        The time to expiration of the option in years.
    multiplier : float
        The multiplier for the option contract.

    Returns
    -------
    BlackScholesGreeksResult
        A named tuple containing the calculated option price, delta, gamma, vega, and theta.
    """

def imply_vol(
    s: float,
    r: float,
    b: float,
    is_call: bool,
    k: float,
    t: float,
    price: float,
) -> float:
    """
    Calculate the implied volatility and Greeks for an option contract.

    Parameters
    ----------
    s : float
        The current price of the underlying asset.
    r : float
        The risk-free interest rate.
    b : float
        The cost of carry of the underlying asset.
    is_call : bool
        Whether the option is a call (True) or a put (False).
    k : float
        The strike price of the option.
    t : float
        The time to expiration of the option in years.
    price : float
        The current market price of the option.
    multiplier : float
        The multiplier for the option contract.

    Returns
    -------
    float
        An implied volatility value.
    """

def imply_vol_and_greeks(
    s: float,
    r: float,
    b: float,
    is_call: bool,
    k: float,
    t: float,
    price: float,
    multiplier: float,
) -> BlackScholesGreeksResult:
    """
    Calculate the implied volatility and Greeks for an option contract.

    Parameters
    ----------
    s : float
        The current price of the underlying asset.
    r : float
        The risk-free interest rate.
    b : float
        The cost of carry of the underlying asset.
    is_call : bool
        Whether the option is a call (True) or a put (False).
    k : float
        The strike price of the option.
    t : float
        The time to expiration of the option in years.
    price : float
        The current market price of the option.
    multiplier : float
        The multiplier for the option contract.

    Returns
    -------
    BlackScholesGreeksResult
        A result containing the calculated implied volatility, option price, delta, gamma, vega, and theta
    """

def refine_vol_and_greeks(
    s: float,
    r: float,
    b: float,
    is_call: bool,
    k: float,
    t: float,
    target_price: float,
    initial_vol: float,
    multiplier: float,
) -> BlackScholesGreeksResult:
    """
    Refine implied volatility using an initial guess and compute greeks.

    This function uses compute_iv_and_greeks which performs a Halley iteration
    to refine the volatility estimate from an initial guess.

    Parameters
    ----------
    s : float
        The current price of the underlying asset.
    r : float
        The risk-free interest rate.
    b : float
        The cost of carry of the underlying asset.
    is_call : bool
        Whether the option is a call (True) or a put (False).
    k : float
        The strike price of the option.
    t : float
        The time to expiration of the option in years.
    target_price : float
        The target market price of the option.
    initial_vol : float
        The initial guess for the volatility.
    multiplier : float
        The multiplier for the option contract.

    Returns
    -------
    BlackScholesGreeksResult
        A result containing the calculated implied volatility, option price, delta, gamma, vega, and theta
    """

class GreeksData(Data):
    instrument_id: InstrumentId
    is_call: bool
    strike: float
    expiry: int
    expiry_in_days: int
    expiry_in_years: float
    multiplier: float
    quantity: float

    underlying_price: float
    interest_rate: float
    cost_of_carry: float

    vol: float
    pnl: float
    price: float
    delta: float
    gamma: float
    vega: float
    theta: float

    itm_prob: float

    def __init__(
        self,
        ts_event: int = 0,
        ts_init: int = 0,
        instrument_id: InstrumentId = ...,
        is_call: bool = True,
        strike: float = 0.0,
        expiry: int = 0,
        expiry_in_days: int = 0,
        expiry_in_years: float = 0.0,
        multiplier: float = 0.0,
        quantity: float = 0.0,
        underlying_price: float = 0.0,
        interest_rate: float = 0.0,
        cost_of_carry: float = 0.0,
        vol: float = 0.0,
        pnl: float = 0.0,
        price: float = 0.0,
        delta: float = 0.0,
        gamma: float = 0.0,
        vega: float = 0.0,
        theta: float = 0.0,
        itm_prob: float = 0.0,
    ): ...
    @classmethod
    def from_delta(
        cls, instrument_id: InstrumentId, delta: float, ts_event: int = 0
    ) -> GreeksData: ...

class PortfolioGreeks(Data):
    pnl: float
    price: float
    delta: float
    gamma: float
    vega: float
    theta: float

    def __init__(
        self,
        ts_event: int = 0,
        ts_init: int = 0,
        pnl: float = 0.0,
        price: float = 0.0,
        delta: float = 0.0,
        gamma: float = 0.0,
        vega: float = 0.0,
        theta: float = 0.0,
    ): ...

class YieldCurveData(Data):
    curve_name: str
    tenors: np.ndarray
    interest_rates: np.ndarray

    def __init__(
        self,
        ts_event: int,
        ts_init: int,
        curve_name: str,
        tenors: np.ndarray,
        interest_rates: np.ndarray,
    ): ...

###################################################################################################
# Live
###################################################################################################

def calculate_reconciliation_price(
    current_position_qty: Decimal,
    current_position_avg_px: Decimal | None,
    target_position_qty: Decimal,
    target_position_avg_px: Decimal | None,
) -> Decimal | None: ...
def adjust_fills_for_partial_window(
    mass_status: ExecutionMassStatus,
    instrument: Any,  # PyO3 InstrumentAny
    tolerance: str | None = None,
) -> tuple[dict[str, OrderStatusReport], dict[str, list[FillReport]]]: ...

###################################################################################################
# Test Kit
###################################################################################################

def ensure_file_exists_or_download_http(
    filepath: str,
    url: str,
    checksums: str | None = None,
    timeout_secs=30,
) -> None: ...

###################################################################################################
# Trading
###################################################################################################

# Converts a UTC timestamp to the local time for the given Forex session.
#
# The `time_now` must be timezone-aware with its tzinfo set to a built-in `datetime.timezone`
# (e.g. `datetime.timezone.utc`). Third-party tzinfo objects (like those from `pytz`) are not supported.
def fx_local_from_utc(session: ForexSession, time_now: dt.datetime) -> str: ...

# Returns the next session start time in UTC.
#
# The `time_now` must be timezone-aware with its tzinfo set to a built-in `datetime.timezone`
# (e.g. `datetime.timezone.utc`). Third-party tzinfo objects (like those from `pytz`) are not supported.
def fx_next_start(session: ForexSession, time_now: dt.datetime) -> dt.datetime: ...

# Returns the next session end time in UTC.
#
# The `time_now` must be timezone-aware with its tzinfo set to a built-in `datetime.timezone`
# (e.g. `datetime.timezone.utc`). Third-party tzinfo objects (like those from `pytz`) are not supported.
def fx_next_end(session: ForexSession, time_now: dt.datetime) -> dt.datetime: ...

# Returns the previous session start time in UTC.
#
# The `time_now` must be timezone-aware with its tzinfo set to a built-in `datetime.timezone`
# (e.g. `datetime.timezone.utc`). Third-party tzinfo objects (like those from `pytz`) are not supported.
def fx_prev_start(session: ForexSession, time_now: dt.datetime) -> dt.datetime: ...

# Returns the previous session end time in UTC.
#
# The `time_now` must be timezone-aware with its tzinfo set to a built-in `datetime.timezone`
# (e.g. `datetime.timezone.utc`). Third-party tzinfo objects (like those from `pytz`) are not supported.
def fx_prev_end(session: ForexSession, time_now: dt.datetime) -> dt.datetime: ...

###################################################################################################
# Analysis
###################################################################################################

class PortfolioAnalyzer:
    def __init__(self) -> None: ...
    def currencies(self) -> list[Currency]: ...
    def get_performance_stats_returns(self) -> dict[str, float]: ...
    def get_performance_stats_pnls(
        self,
        currency: Currency | None = None,
        unrealized_pnl: Money | None = None,
    ) -> dict[str, float]: ...
    def get_performance_stats_general(self) -> dict[str, float]: ...
    def add_return(self, timestamp: int, value: float) -> None: ...
    def reset(self) -> None: ...

class SharpeRatio:
    def __init__(self, period: int = 252) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class SortinoRatio:
    def __init__(self, period: int = 252) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class ReturnsVolatility:
    def __init__(self, period: int = 252) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class ReturnsAverage:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class ReturnsAverageLoss:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class ReturnsAverageWin:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class ProfitFactor:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class RiskReturnRatio:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class CAGR:
    def __init__(self, period: int | None = None) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class CalmarRatio:
    def __init__(self, period: int | None = None) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class MaxDrawdown:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_returns(self, returns: dict[int, float]) -> float | None: ...

class WinRate:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class Expectancy:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class AvgWinner:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class MaxWinner:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class MinWinner:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class AvgLoser:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class MaxLoser:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class MinLoser:
    def __init__(self) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_realized_pnls(self, realized_pnls: list[float]) -> float | None: ...

class LongRatio:
    def __init__(self, precision: int = 2) -> None: ...
    @property
    def name(self) -> str: ...
    def calculate_from_positions(self, positions: list[Position]) -> float | None: ...

###################################################################################################
# Binance
###################################################################################################

class BinanceProductType(Enum):
    SPOT = "SPOT"
    MARGIN = "MARGIN"
    USD_M = "USD_M"
    COIN_M = "COIN_M"
    OPTIONS = "OPTIONS"

class BinanceEnvironment(Enum):
    MAINNET = "MAINNET"
    TESTNET = "TESTNET"

class BinanceSpotHttpClient:
    def __init__(
        self,
        environment: BinanceEnvironment = ...,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        recv_window: int | None = None,
        timeout_secs: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @property
    def schema_id(self) -> int: ...
    @property
    def schema_version(self) -> int: ...
    async def ping(self) -> None: ...
    async def server_time(self) -> int: ...
    async def request_instruments(self) -> list[Instrument]: ...
    async def request_trades(
        self,
        instrument_id: InstrumentId,
        limit: int | None = None,
    ) -> list[TradeTick]: ...
    async def request_order_status(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId | None = None,
        client_order_id: ClientOrderId | None = None,
    ) -> OrderStatusReport: ...
    async def request_order_status_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        open_only: bool = False,
        limit: int | None = None,
    ) -> list[OrderStatusReport]: ...
    async def request_fill_reports(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId | None = None,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[FillReport]: ...
    async def request_bars(
        self,
        bar_type: BarType,
        start: dt.datetime | None = None,
        end: dt.datetime | None = None,
        limit: int | None = None,
    ) -> list[Bar]: ...
    async def submit_order(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        price: Price | None = None,
        trigger_price: Price | None = None,
        post_only: bool = False,
    ) -> OrderStatusReport: ...
    async def modify_order(
        self,
        account_id: AccountId,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId,
        client_order_id: ClientOrderId,
        order_side: OrderSide,
        order_type: OrderType,
        quantity: Quantity,
        time_in_force: TimeInForce,
        price: Price | None = None,
    ) -> OrderStatusReport: ...
    async def cancel_order(
        self,
        instrument_id: InstrumentId,
        venue_order_id: VenueOrderId | None = None,
        client_order_id: ClientOrderId | None = None,
    ) -> VenueOrderId: ...
    async def cancel_all_orders(
        self,
        instrument_id: InstrumentId,
    ) -> list[VenueOrderId]: ...

class BinanceFuturesHttpClient:
    def __init__(
        self,
        product_type: BinanceProductType,
        environment: BinanceEnvironment = ...,
        api_key: str | None = None,
        api_secret: str | None = None,
        base_url: str | None = None,
        recv_window: int | None = None,
        timeout_secs: int | None = None,
        proxy_url: str | None = None,
    ) -> None: ...
    @property
    def product_type(self) -> BinanceProductType: ...
    async def server_time(self) -> int: ...
    async def request_instruments(self) -> list[Instrument]: ...
    async def create_listen_key(self) -> str: ...
    async def keepalive_listen_key(self, listen_key: str) -> None: ...
    async def close_listen_key(self, listen_key: str) -> None: ...
    async def query_hedge_mode(self) -> bool: ...
    async def query_account(self) -> dict[str, Any]: ...

class BinanceSpotWebSocketClient:
    def __init__(
        self,
        url: str | None = None,
        api_key: str | None = None,
        api_secret: str | None = None,
        heartbeat: int | None = None,
    ) -> None: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def subscription_count(self) -> int: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    async def connect(self, callback: Callable) -> None: ...
    async def close(self) -> None: ...
    async def subscribe(self, streams: list[str]) -> None: ...
    async def unsubscribe(self, streams: list[str]) -> None: ...

class BinanceFuturesWebSocketClient:
    def __init__(
        self,
        product_type: BinanceProductType,
        environment: BinanceEnvironment,
        api_key: str | None = None,
        api_secret: str | None = None,
        url_override: str | None = None,
        heartbeat: int | None = None,
    ) -> None: ...
    @property
    def product_type(self) -> BinanceProductType: ...
    def is_active(self) -> bool: ...
    def is_closed(self) -> bool: ...
    def subscription_count(self) -> int: ...
    def cache_instrument(self, instrument: Instrument) -> None: ...
    async def connect(self, callback: Callable) -> None: ...
    async def close(self) -> None: ...
    async def subscribe(self, streams: list[str]) -> None: ...
    async def unsubscribe(self, streams: list[str]) -> None: ...
